use assert_cmd::Command;

const YARN_LOCK_V6_WITH_DEPS: &str = r#"# This file is generated by running "yarn install" inside your project.
# Manual changes might be lost - proceed with caution!

__metadata:
  version: 6
  cacheKey: 8

"foo@workspace:.":
  version: 0.0.0-use.local
  resolution: "foo@workspace:."
  dependencies:
    foolib: 1.2.3 || ^2.0.0
  languageName: unknown
  linkType: soft

"foolib@npm:1.2.3 || ^2.0.0":
  version: 2.0.0
  resolution: "foolib@npm:2.0.0"
  checksum: 123061e52a0b3792c6a0472bf48ca6c337ccb58e92261049e7727a12c326b9627537e2ef8cb4453354d02c763b87c8b516f4eedfad99945c308927285bbc12ba
  languageName: node
  linkType: hard
"#;

const YARN_LOCK_V6_ONLY_DIRECT_DEPS: &str = r#"# This file is generated by running "yarn install" inside your project.
# Manual changes might be lost - proceed with caution!

__metadata:
  version: 6
  cacheKey: 8

"foolib@npm:1.2.3 || ^2.0.0":
  version: 2.0.0
  resolution: "foolib@npm:2.0.0"
  checksum: 123061e52a0b3792c6a0472bf48ca6c337ccb58e92261049e7727a12c326b9627537e2ef8cb4453354d02c763b87c8b516f4eedfad99945c308927285bbc12ba
  languageName: node
  linkType: hard
"#;

#[test]
fn it_fails_and_display_the_help_when_invoked_without_args() {
    let mut cmd = Command::cargo_bin(env!("CARGO_PKG_NAME")).unwrap();

    let assert = cmd.assert().failure().code(1);
    let stdout = std::str::from_utf8(&assert.get_output().stdout).unwrap();
    assert!(stdout.contains("Usage:"), "Output was: {}", stdout);
}

#[test]
fn it_succeed_and_display_the_help_when_invoked_with_help_option() {
    let mut cmd = Command::cargo_bin(env!("CARGO_PKG_NAME")).unwrap();

    let assert = cmd.args(&["-h"]).assert().success();
    let stdout = std::str::from_utf8(&assert.get_output().stdout).unwrap();
    assert!(stdout.contains("Usage:"), "Output was: {}", stdout);
}

#[test]
fn it_displays_the_version() {
    let mut cmd = Command::cargo_bin(env!("CARGO_PKG_NAME")).unwrap();

    let assert = cmd.args(&["-V"]).assert().success();
    let stdout = std::str::from_utf8(&assert.get_output().stdout).unwrap();
    assert_eq!(
        concat!(env!("CARGO_PKG_NAME"), " ", env!("CARGO_PKG_VERSION"), "\n"),
        stdout
    );
}

#[test]
fn it_finds_a_package_with_range() {
    let mut cmd = Command::cargo_bin(env!("CARGO_PKG_NAME")).unwrap();

    let assert = cmd
        .args(&["foolib@1.2.3 || ^2.0.0"])
        .write_stdin(YARN_LOCK_V6_WITH_DEPS)
        .assert();

    assert.success().stdout(
        r#"foo@.
   foolib@1.2.3 || ^2.0.0
"#,
    );
}

#[test]
fn it_finds_a_package_without_range() {
    let mut cmd = Command::cargo_bin(env!("CARGO_PKG_NAME")).unwrap();

    let assert = cmd
        .args(&["foolib"])
        .write_stdin(YARN_LOCK_V6_WITH_DEPS)
        .assert();

    assert.success().stdout(
        r#"foo@.
   foolib@1.2.3 || ^2.0.0
"#,
    );
}

#[test]
fn it_finds_a_package_in_a_yarn_lock_with_only_direct_deps() {
    let mut cmd = Command::cargo_bin(env!("CARGO_PKG_NAME")).unwrap();

    let assert = cmd
        .args(&["foolib"])
        .write_stdin(YARN_LOCK_V6_ONLY_DIRECT_DEPS)
        .assert();

    assert.success().stdout("foolib@1.2.3 || ^2.0.0\n");
}
