//! Ring buffer storage

use core::{
	cell::UnsafeCell,
	mem::MaybeUninit,
	ptr::NonNull,
};

/// Trait for ring buffer storage.
///
/// Pluggable interface for chosing how to store
///
/// This only provides storage to the ring buffer - the ring buffer code will
/// take care of reading, writing, and dropping elements.
///
/// # Safety
///
/// The storage itself must not access the buffer, outside of creation and
/// dropping. It should only handle allocation and deallocation. Additionally,
/// the buffer may hold uninitialized data, so it should be stored through a
/// raw pointer or `UnsafeCell<MaybeUninit<_>>`.
pub unsafe trait Storage<T> {
	/// Gets a pointer to the storage.
	fn ptr(&mut self) -> NonNull<T>;
	/// Gets the size of the storage
	fn len(&self) -> usize;
}

#[cfg(feature = "alloc")]
mod global_alloc {
	use core::alloc::Layout;

	use ::alloc::alloc;

	use super::*;

	/// Default storage. Allocates the buffer from the global allocator.
	pub struct GlobalAllocStorage<T>(NonNull<T>, usize);
	impl<T> GlobalAllocStorage<T> {
		pub fn new(size: usize) -> Self {
			let layout = Layout::array::<T>(size).unwrap();
			let p = NonNull::new(unsafe { alloc::alloc(layout) })
				.unwrap_or_else(|| alloc::handle_alloc_error(layout));
			Self(p.cast(), size)
		}
	}
	unsafe impl<T> Storage<T> for GlobalAllocStorage<T> {
		fn ptr(&mut self) -> NonNull<T> {
			self.0
		}

		fn len(&self) -> usize {
			self.1
		}
	}
	impl<T> core::ops::Drop for GlobalAllocStorage<T> {
		fn drop(&mut self) {
			let layout = Layout::array::<T>(self.1).unwrap();
			unsafe { alloc::dealloc(self.0.cast().as_mut(), layout) }
		}
	}
}
#[cfg(feature = "alloc")]
pub use global_alloc::*;

/// Storage backed by a fixed length array.
/// 
/// The array will be embedded in the buffer metadata.
/// 
/// Useful if the buffer doesn't need variable size, and for embedded.
pub struct ArrayStorage<T, const D: usize>(UnsafeCell<MaybeUninit<[T; D]>>);
impl<T, const D: usize> ArrayStorage<T, D> {
	/// Creates the storage.
	pub fn new() -> Self {
		Self(UnsafeCell::new(MaybeUninit::uninit()))
	}
}
unsafe impl<T, const D: usize> Storage<T> for ArrayStorage<T, D> {
	fn ptr(&mut self) -> NonNull<T> {
		let mibuffer = unsafe { &mut *self.0.get() };
		NonNull::new(mibuffer.as_mut_ptr().cast()).unwrap()
	}

	fn len(&self) -> usize {
		D
	}
}
