//! Module for the xlib connection
//!

use x11_dl::{error, xlib};

use std::ptr;

/// main struct of the module
pub struct Xlib {
    connection: xlib::Xlib,
}

impl Xlib {
    /// # Open
    /// First function that should be called to interact with this library.
    ///
    /// ## Errors
    /// * TODO find out how to link to the Xlib OpenErrors here.
    pub fn open() -> Result<Self> {
        // open the connection to the library.
        // if it fails we just pass up the error.
        let connection = xlib::Xlib::open()?;
        Ok(Self { connection })
    }

    /// # open display
    ///
    /// ## Safety
    /// TODO
    pub unsafe fn open_display(&self, dpy: Option<&str>) -> Result<*mut xlib::Display> {
        let res = match dpy {
            Some(dp) => (self.connection.XOpenDisplay)(dp.as_ptr() as *const i8),
            None => (self.connection.XOpenDisplay)(ptr::null()),
        };
        if res.is_null() {
            Err(Error::DisplayOpenError(
                dpy.unwrap_or("[DEFAULT]").to_owned(),
            ))
        } else {
            Ok(res)
        }
    }
}

#[derive(Debug, thiserror::Error)]
/// # Errors
/// All the errors that the Xlib struct can provide.
pub enum Error {
    #[error("Failed to open XDisplay with name: {0}")]
    DisplayOpenError(String),
    #[error(transparent)]
    /// Just wrapping the x11_dl already provided error.
    XlibOpenError(#[from] error::OpenError),
}

/// my own error type for simplicity
type Result<T> = std::result::Result<T, Error>;
