/// Driver for trees

// Find the string value of the Node
fn node_to_string(node: &RcNode<NodeDefn>) -> String {
  let d = node.data();

  match d.nodetype {
      NodeType::Document => {
        if node.has_no_child() {
	  String::new()
	} else {
	  node.iter_rc().fold(String::new(), |s,c| s + &node_to_string(&c))
	}
      }
      NodeType::Element => {
        if node.has_no_child() {
	  "".to_string()
	} else {
	  node.iter_rc().fold(String::new(), |s,c| s + &node_to_string(&c))
	}
      }
      NodeType::Text => {
        String::from(d.value.as_ref().unwrap())
      }
      NodeType::Attribute |
      NodeType::Comment |
      NodeType::ProcessingInstruction => {
        "".to_string()
      }
  }
}
fn node_to_xml(node: &RcNode<NodeDefn>) -> String {
  let d = node.data();

  match d.nodetype {
      NodeType::Document => {
        if node.has_no_child() {
	  String::new()
	} else {
	  node.iter_rc().fold(String::new(), |s,c| s + &node_to_xml(&c))
	}
      }
      NodeType::Element => {
        if node.has_no_child() {
	  format!("<{}/>", d.name.as_ref().unwrap())
	} else {
	  // TODO: attributes
	  format!("<{}>{}</{}>", d.name.as_ref().unwrap(), node.iter_rc().fold(String::new(), |s,c| s + &node_to_xml(&c)), d.name.as_ref().unwrap())
	}
      }
      NodeType::Text => {
        String::from(d.value.as_ref().unwrap())
      }
      NodeType::Attribute => {
        let mut r = String::new();
        r.push_str(d.name.as_ref().unwrap().as_str());
        r.push_str("='");
        r.push_str(d.value.as_ref().unwrap().as_str());
        r.push_str("'");
        // TODO: delimiters, escaping
	r
      }
      NodeType::Comment => {
        let mut r = String::new();
        r.push_str("<!--");
        r.push_str(d.value.as_ref().unwrap().as_str());
        r.push_str("-->");
	r
      }
      NodeType::ProcessingInstruction => {
        let mut r = String::new();
        r.push_str("<?");
        r.push_str(d.name.as_ref().unwrap().as_str());
        r.push_str(" ");
        r.push_str(d.value.as_ref().unwrap().as_str());
        r.push_str("?>");
	r
      }
  }
}
fn node_to_json(node: &RcNode<NodeDefn>) -> String {
  let d = node.data();

  match d.nodetype {
      NodeType::Document => {
        if node.has_no_child() {
	  "{}".to_string()
	} else {
	  let mut r = String::from("{");
	  r.push_str(node.iter_rc().fold(String::new(), |s,c| s + &node_to_json(&c)).as_str());
	  r.push('}');
	  r
	}
      }
      NodeType::Element => {
        if node.has_no_child() {
	  format!("\"{}\": \"\"", d.name.as_ref().unwrap())
	} else {
	  format!("\"{}\": {}\n", d.name.as_ref().unwrap(), node.iter_rc().fold(String::new(), |s,c| s + &node_to_json(&c)))
	}
      }
      NodeType::Text => {
        format!("\"{}\"", String::from(d.value.as_ref().unwrap()))
      }
      NodeType::Attribute => {
        "".to_string()
      }
      NodeType::Comment => {
        "".to_string()
      }
      NodeType::ProcessingInstruction => {
        "".to_string()
      }
  }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn node_document() {
        RcNode::from(Tree::new(NodeDefn::new(NodeType::Document)));
        assert!(true)
    }
    #[test]
    fn node_element() {
        let d = RcNode::from(Tree::new(NodeDefn::new(NodeType::Document)));
        let e = Tree::new(NodeDefn::new(NodeType::Element).set_name("Test".to_string()));
	d.push_front(e);
        assert_eq!(node_to_string(&d), "");
        assert_eq!(node_to_xml(&d), "<Test/>");
        assert_eq!(node_to_json(&d), "{\"Test\": \"\"}")
    }
    #[test]
    fn node_text() {
        let d = RcNode::from(Tree::new(NodeDefn::new(NodeType::Document)));
        let mut e = Tree::new(NodeDefn::new(NodeType::Element).set_name("Test".to_string()));
        let t = Tree::new(NodeDefn::new(NodeType::Text).set_value("Test text".to_string()));
	e.push_front(t);
	d.push_front(e);
        assert_eq!(node_to_string(&d), "Test text");
        assert_eq!(node_to_xml(&d), "<Test>Test text</Test>")
    }
    #[test]
    fn item_node_to_string() {
        let d = RcNode::from(Tree::new(NodeDefn::new(NodeType::Document)));
        let mut e = Tree::new(NodeDefn::new(NodeType::Element).set_name("Test".to_string()));
        let t = Tree::new(NodeDefn::new(NodeType::Text).set_value("Test text".to_string()));
	e.push_front(t);
	d.push_front(e);
	let i = Item::Node(d);
        assert_eq!(i.to_string(), "Test text");
        assert_eq!(i.to_xml(), "<Test>Test text</Test>")
    }


}

