//! # xor_rand
//!
//! 'xor_rand' is an implimentation for arbitrary state size
//! pseudo-random number generator (PRNG), primarily using XOR and ADD
//! for pseudo-random counter based hash randomness
use std::num::Wrapping;

pub struct XorRand {
    state: Box<[u8]>,
}

impl XorRand {
    /// Provides a new or default implimentation of the PRNG
    pub fn new(state_length: usize) -> XorRand {
        XorRand {
            state: vec![0; state_length].into_boxed_slice(),
        }
    }
    /// Returns the state length of the prng
    pub fn state_len(&self) -> usize {
        self.state.len()
    }
    /// Seeds the PRNG with whatever data you wish,
    /// it changes the state length of the struct to match the seed.
    /// can be used with [data].to_be_bytes() or [data].to_le_bytes()
    pub fn seed(&mut self, seed: &[u8]) {
        self.state = seed.to_owned().into_boxed_slice()
    }
    /// Generates one pseudo random byte
    /// Updated in 1.3100 for better mixing
    /// Updated in 1.3200 for better distribution
    /// Updatef in 1.3300 for better uniformity
    pub fn next_byte(&mut self) -> u8 {
        match self.state.get(0) {
            Some(first_elem) => {
                let mut mixed = *first_elem;
                for (i, elem) in self.state.iter().rev().enumerate() {
                    let (i, elem) = (Wrapping(i as u8), Wrapping(*elem));
                    mixed = (Wrapping(mixed << 3 | mixed >> 5) ^ i).0;
                    mixed ^= mixed >> 3;
                    mixed = (Wrapping(mixed << 5 | mixed >> 3) ^ elem).0;
                    mixed ^= mixed >> 3;
                    mixed = (Wrapping(mixed << 5 | mixed >> 3) + Wrapping(17u8) * i).0;
                    mixed ^= mixed >> 3;
                    mixed = (Wrapping(mixed << 3 | mixed >> 5) - Wrapping(17u8) * elem).0;
                    mixed ^= mixed >> 3;
                }
                let mut carry: u16 = 1;
                for elem in self.state.iter_mut() {
                    carry += *elem as u16;
                    *elem = carry as u8;
                    carry = carry >> 8;
                }
                mixed
            }
            _ => 0,
        }
    }
    /// Generates a static array of pseudo random bytes
    /// use with [type].from_be_bytes() or [type].from_le_bytes()
    /// to convert between types easily
    pub fn next_bytes<const LEN: usize>(&mut self) -> [u8; LEN] {
        let mut out = [0; LEN];
        //run the mixing step for every output byte
        for state in out.iter_mut() {
            *state = self.next_byte();
        }
        //return permuted byte
        out
    }
}

impl std::io::Read for XorRand {
    /// Preserves the index and state of the PRNG,
    /// can be used to read from the PRNG output arbitrarily.

    fn read(&mut self, buf: &mut [u8]) -> std::io::Result<usize> {
        for byte in buf.iter_mut() {
            *byte = self.next_byte();
        }
        Ok(buf.len())
    }
}

impl Iterator for XorRand {
    type Item = u8;
    /// Turns the XorRand crate into a structure that is iterator friendly.
    /// moves internal state, so be careful when calling to not clone the state

    fn next(&mut self) -> Option<Self::Item> {
        Some(self.next_byte())
    }
}

impl Default for XorRand {
    /// Initialize with a blank state at 64 bits, or 8 bytes.
    fn default() -> Self {
        XorRand {
            state: vec![0; 8].into_boxed_slice(),
        }
    }
}
