//! # xor_rand
//!
//! 'xor_rand' is an implimentation for arbitrary state size
//! pseudo-random number generator (PRNG)

//Used to disable the arithmatic overflow protection in the std, for using larger STATE_BYTE sizes.
#![allow(arithmetic_overflow)]
#![cfg_attr(not(feature = "std"), no_std)]
pub struct XorRand<const STATE_BYTES: usize> {
    //state consists of the working permuted byte, and the state array
    state: [u8; STATE_BYTES],
    permute: u8,
}
impl<const STATE_BYTES: usize> XorRand<STATE_BYTES> {
    pub fn new(seed: [u8; STATE_BYTES]) -> XorRand<STATE_BYTES> {
        XorRand {
            //Construct a new state, with the given state and initial permuted byte of 0x55
            state: seed,
            permute: 0x55,
        }
    }
    /// Generates one pseudo random byte
    ///
    /// # Examples:
    /// ```
    /// // Create a new instance of the XorRand struct
    /// let mut prng = xor_rand::XorRand::new([0_u8; 4]);
    /// // Fetch the next byte out of the mutable instance
    /// let output = prng.next_byte();
    /// // Do what you want with the output
    /// match output % 4 {
    ///     0 => println!("I choose 0!"),
    ///     1 => println!("I choose 1!"),
    ///     2 => println!("I choose 2!"),
    ///     3 => println!("I choose 3!!!"),
    ///     _ => unreachable!()
    /// }
    /// ```
    pub fn next_byte(&mut self) -> u8 {
        for (num, byte) in self.state.iter().enumerate() {
            //Fairly good (To my knowledge) method of generating PRNG mixed bytes
            self.permute = ((self.permute >> 3) | (self.permute << 5)) ^ byte;
            self.permute ^= self.permute << 4;
            self.permute = ((self.permute << 3) | (self.permute >> 5)) ^ num as u8;
            self.permute ^= self.permute << 4;
            self.permute =
                (((self.permute << 3) | (self.permute >> 5)) as u16 + *byte as u16) as u8;
            self.permute ^= self.permute << 4;
            self.permute = (((self.permute >> 3) | (self.permute << 5)) as u16 + num as u16) as u8;
            self.permute ^= self.permute << 4;
        }
        //increment the counter, little endian
        let mut carry: u16 = 1;
        let mut index: usize = 0;
        loop {
            carry += self.state[index] as u16;
            self.state[index] = carry as u8;
            carry >>= 8;
            index += 1;
            //Stop the increment loop if you are adding 0, or finished
            if index >= STATE_BYTES || carry == 0 {
                break;
            }
        }
        self.permute
    }
    /// Generates a static array of pseudo random bytes
    ///
    /// # Examples:
    /// ```
    /// // Create a new instance of the XorRand struct
    /// let mut prng = xor_rand::XorRand::new([0_u8; 4]);
    /// // Fetch the next 8 bytes out of the instance
    /// let output = prng.next_bytes::<8>();
    /// // Do what you want with the output
    /// println!("Here is your 64 bit random number:");
    /// for byte in output {
    ///     print!("{:02X}", byte);
    /// }
    /// println!();
    /// ```
    pub fn next_bytes<const LEN: usize>(&mut self) -> [u8; LEN] {
        //Initialize the output array to 0x55 per byte for LEN length
        let mut out = [0x55; LEN];
        //run the mixing step for every output byte
        for state in out.iter_mut() {
            //mix the index of the state, as well as the value of the state into the permuted byte
            *state = self.next_byte();
        }
        //return permuted byte
        out
    }
}

#[cfg(feature = "std")]
impl<const STATE_BYTES: usize> std::io::Read for XorRand<STATE_BYTES> {
    /// Preserves the index and state of the PRNG,
    /// can be used to read from the PRNG output arbitrarily.
    ///
    /// # Examples:
    /// ```
    /// let mut prng = xor_rand::XorRand::new([0u8; 16]);
    /// let mut random_nums = [0_u8; 16];
    /// loop {
    ///     prng.read_exact(&mut random_nums).unwrap();
    ///     let string_rep = random_nums
    ///         .iter()
    ///         .map(|val| format!("{:02X}", val))
    ///         .collect::<String>();
    ///     println!("Some randomness: {}", string_rep);
    /// }
    /// ```
    fn read(&mut self, buf: &mut [u8]) -> std::io::Result<usize> {
        for byte in buf.iter_mut() {
            *byte = self.next_byte();
        }
        Ok(buf.len())
    }
}

impl<const STATE_BYTES: usize> Iterator for XorRand<STATE_BYTES> {
    type Item = u8;
    /// Turns the XorRand crate into a structure that is iterator friendly.
    ///
    /// Take care when using this iterator, creating an iterator moves
    /// the state of the XorRand instance, which can be undesirable for reuse.
    /// Use the std::io::Read attribute for constant reuse.
    ///
    /// # Examples:
    /// ```
    /// use std::{io::{Write, stdout}, thread::sleep, time::Duration};
    /// let prng = xor_rand::XorRand::new([0_u8; 16]);
    /// for byte in prng.take(16777216) {
    ///     for bits in 0..4 {
    ///         match (byte >> (bits * 2)) & 0b11 {
    ///             0 => print!(" "),
    ///             1 => print!("▄"),
    ///             2 => print!("▀"),
    ///             3 => print!("█"),
    ///             _ => unreachable!()
    ///         }
    ///         stdout().flush().unwrap();
    ///         sleep(Duration::from_millis(10));
    ///     }
    /// }
    /// ```
    fn next(&mut self) -> Option<Self::Item> {
        Some(self.next_byte())
    }
}
