//! # xor_rand
//!
//! 'xor_rand' is an implimentation for arbitrary state size
//! pseudo-random number generator (PRNG)
pub struct XorRand<const STATE_BYTES: usize> {
    //state consists of the working permuted byte, and the state array
    state: [u8; STATE_BYTES],
    permute: u8,
}
impl<const STATE_BYTES: usize> XorRand<STATE_BYTES> {
    /*
        Construct a new XorRand instance, with a given initial state (seed).
        The STATE_BYTES constant generic represents the number of bytes in the state,
        and correlates roughly to how long the period of the PRNG is, as well as the quality of
        the mixed and pseudo random numbers produced.
    */
    pub fn new(seed: [u8; STATE_BYTES]) -> XorRand<STATE_BYTES> {
        XorRand {
            //Construct a new state, with the given state and initial permuted byte of 0x55
            state: seed,
            permute: 0x55,
        }
    }
    /*
        Recieve an array of the produced values from the function, on the stack.
        This function should produce a uniform output, such that concatenating 256 calls
        of the function with LEN = 1, producing the same as 1 call of LEN = 256.

        The larger the constant generic STATE_BYTES happens to be, the slower this function
        will be to call. This function is not indended to be thread safe.
    */
    /// Generates a static array of pseudo random bytes
    ///
    /// # Examples:
    /// ```
    /// let mut prng = xor_rand::XorRand::new([0_u8; 4]);
    /// let output = prng.next_bytes::<8>();
    /// dbg!(output);
    /// ```
    pub fn next_bytes<const LEN: usize>(&mut self) -> [u8; LEN] {
        //Initialize the output array to 0x55 per byte for LEN length
        let mut out = [0x55; LEN];
        //run the mixing step for every output byte
        for state in out.iter_mut() {
            //mix the index of the state, as well as the value of the state into the permuted byte
            for (num, byte) in self.state.iter().enumerate() {
                //Fairly good (To my knowledge) method of generating PRNG mixed bytes
                self.permute = ((self.permute >> 3) | (self.permute << 5)) ^ byte;
                self.permute ^= self.permute << 4;
                self.permute = ((self.permute << 3) | (self.permute >> 5)) ^ num as u8;
                self.permute ^= self.permute << 4;
                self.permute = ((self.permute << 3) | (self.permute >> 5)) + byte;
                self.permute ^= self.permute << 4;
                self.permute = ((self.permute >> 3) | (self.permute << 5)) + num as u8;
                self.permute ^= self.permute << 4;
            }
            *state = self.permute;
            //increment the counter, little endian
            let mut carry: u16 = 1;
            let mut index: usize = 0;
            loop {
                carry += self.state[index] as u16;
                self.state[index] = carry as u8;
                carry >>= 8;
                index += 1;
                //Stop the increment loop if you are adding 0, or finished
                if index >= STATE_BYTES || carry == 0 {
                    break;
                }
            }
        }
        //return permuted byte
        out
    }
}
