use std::fmt::{Error, Write};

use crate::{AttValueSafe, NameSafe, PcdataSafe};

/// Encapsulates a `std::fmt::Write` target and prevents XML
/// injections through marker traits and typestates.
pub struct XmlWriter<'a>(&'a mut dyn Write);

impl<'a> XmlWriter<'a> {
    /// Creates a new `XmlWriter` from a `std::fmt::Write` implementation.
    pub fn new(writer: &mut impl Write) -> XmlWriter {
        XmlWriter(writer)
    }

    /// Writes a start tag.
    pub fn write_start_tag(&mut self, name: impl NameSafe) -> Result<(), Error> {
        write!(self.0, "<{}>", name)
    }

    /// Consumes the writer, opens a start tag and returns an [`AttWriter`].
    pub fn open_start_tag(self, name: impl NameSafe) -> Result<AttWriter<'a>, Error> {
        write!(self.0, "<{}", name)?;
        Ok(AttWriter(self))
    }

    /// Writes an end tag.
    pub fn write_end_tag(&mut self, name: impl NameSafe) -> Result<(), Error> {
        write!(self.0, "</{}>", name)
    }

    /// Writes some PCDATA.
    pub fn write(&mut self, value: impl PcdataSafe) -> Result<(), Error> {
        self.0.write_fmt(format_args!("{}", value))
    }

    /// Duplicates the `XmlWriter`, which is useful to delegate part of the XML
    /// generation to other functions (since `open_start_tag` requires `self`).
    pub fn duplicate(&mut self) -> XmlWriter {
        XmlWriter(self.0)
    }
}

/// An XML attribute writer returned by [`XmlWriter::open_start_tag`].
pub struct AttWriter<'a>(XmlWriter<'a>);

impl<'a> AttWriter<'a> {
    /// Writes an attribute. Avoid writing two attributes with the same name or
    /// your XML will be invalid.
    pub fn write_attr(
        &mut self,
        name: impl NameSafe,
        value: impl AttValueSafe,
    ) -> Result<(), Error> {
        write!(self.0 .0, " {}=\"{}\"", name, value)
    }

    /// Writes the given attribute and returns the Writer to allow method chaining.
    /// Avoid writing two attributes with the same name or your XML will be
    /// invalid.
    pub fn attr(mut self, name: impl NameSafe, value: impl AttValueSafe) -> Result<Self, Error> {
        self.write_attr(name, value)?;
        Ok(self)
    }

    /// Consumes the writer, writes a key and returns an [`AttValueWriter`].
    pub fn key(self, name: impl NameSafe) -> Result<AttValueWriter<'a>, Error> {
        write!(self.0 .0, " {}=\"", name)?;
        Ok(AttValueWriter(self))
    }

    /// Closes the opening tag, returning an XmlWriter you can use to write the element content.
    /// To produce a valid XML document you will need to close the tag after the content.
    pub fn close(self) -> Result<XmlWriter<'a>, Error> {
        self.0 .0.write_char('>')?;
        Ok(self.0)
    }

    /// Closes the element with a self-closing tag. Returns the XmlWriter so you
    /// can continue writing the document.
    pub fn close_empty(self) -> Result<XmlWriter<'a>, Error> {
        self.0 .0.write_str("/>")?;
        Ok(self.0)
    }
}

/// An XML attribute value writer returned by [`AttWriter::key`].
pub struct AttValueWriter<'a>(AttWriter<'a>);

impl<'a> AttValueWriter<'a> {
    /// Write part of the attribute value.
    pub fn write(&mut self, value: impl AttValueSafe) -> Result<(), Error> {
        self.0 .0 .0.write_fmt(format_args!("{}", value))
    }

    /// Ends the quoted value, returning an AttWriter.
    pub fn end(self) -> Result<AttWriter<'a>, Error> {
        self.0 .0 .0.write_char('"')?;
        Ok(self.0)
    }
}

#[cfg(test)]
mod tests {
    use crate::{format_att_value, format_text, AttValueSafe, PcdataSafe, XmlWriter};

    #[test]
    fn open_tag_write_and_close_tag() {
        let mut out = String::new();
        let writer = XmlWriter::new(&mut out);
        let mut writer = writer
            .open_start_tag("hello-world")
            .unwrap()
            .attr("id", 333)
            .unwrap()
            .close()
            .unwrap();
        writer.write("some text").unwrap();
        writer.write_end_tag("hello-world").unwrap();
        assert_eq!(out, "<hello-world id=\"333\">some text</hello-world>");
    }

    #[test]
    fn test_duplicate() {
        fn subroutine(mut x: XmlWriter) {
            x.write("hello from subroutine").unwrap();
        }

        let mut out = String::new();
        let mut writer = XmlWriter::new(&mut out);
        writer.write_start_tag("hello").unwrap();
        subroutine(writer.duplicate());
        writer.write_end_tag("hello").unwrap();
        assert_eq!(out, "<hello>hello from subroutine</hello>");
    }

    #[test]
    fn test_format_text() {
        struct CustomPcdata;

        impl PcdataSafe for CustomPcdata {}

        impl std::fmt::Display for CustomPcdata {
            fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
                f.write_str("hello from display")
            }
        }

        let mut out = String::new();
        let mut writer = XmlWriter::new(&mut out);
        writer
            .write(format_text!("> {} < {}", CustomPcdata, "test"))
            .unwrap();
        assert_eq!(out, "> hello from display < test");
    }

    #[test]
    fn test_format_att_value() {
        struct CustomAttValue;

        impl AttValueSafe for CustomAttValue {}

        impl std::fmt::Display for CustomAttValue {
            fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
                f.write_str("hello from display")
            }
        }

        let mut out = String::new();
        let writer = XmlWriter::new(&mut out);
        writer
            .open_start_tag("test")
            .unwrap()
            .attr(
                "foo",
                format_att_value!("> {} < {}", CustomAttValue, "test"),
            )
            .unwrap()
            .close()
            .unwrap();
        assert_eq!(out, "<test foo=\"> hello from display < test\">");
    }
}
