use std::fmt::Display;

use crate::AttWriter;

/// Types whose `Display` implementation can be safely used as an XML name.
pub trait NameSafe: Display {}

/// Types whose `Display` implementation can be safely embedded in
/// double-quoted XML attribute values. Literal `"` and `&` characters must be
/// escaped as `&quot;` and `&amp;` respectively.
pub trait AttValueSafe: Display {}

/// Types whose `Display` implementation can be safely embedded between XML
/// tags.  Literal `<` and `&` characters must be escaped as `&lt;` and `&amp;`
/// respectively.
pub trait PcdataSafe: Display {}

impl NameSafe for &'static str {}
impl AttValueSafe for &'static str {}
impl PcdataSafe for &'static str {}

// All type parameters have an implicit bound of `Sized`. We remove this
// bound since we also want NameSafe to be impled for &dyn Trait.
impl<T: NameSafe + ?Sized> NameSafe for &T {}
impl<T: NameSafe + ?Sized> AttValueSafe for &T {}
impl<T: NameSafe + ?Sized> PcdataSafe for &T {}

macro_rules! primitive_impls {
    ($($type: ty),+) => {
        $(
            impl AttValueSafe for $type {}
            impl PcdataSafe for $type {}
        )+
    };
}

primitive_impls!(
    bool, char, f32, f64, i8, i16, i32, i64, i128, isize, u8, u16, u32, u64, u128, usize
);

/// The `tag!` macro expects its first argument to implement this trait, see
/// [Tag constructors](crate::with_writer#tag-constructors).
pub trait TagConstructor {
    /// Returns the tag name of the element.
    fn tag_name(&self) -> &dyn NameSafe;

    /// Lets implementations write an initial set of attributes for the newly created tag.
    fn attributes<'a>(&self, writer: AttWriter<'a>) -> Result<AttWriter<'a>, std::fmt::Error> {
        Ok(writer)
    }
}

impl<N: NameSafe> TagConstructor for N {
    fn tag_name(&self) -> &dyn NameSafe {
        self
    }
}
