use std::{borrow::Cow, fmt::Display};

use crate::{AttValueSafe, PcdataSafe};

/// XML escape an untrusted string to make it `AttValueSafe`.
pub fn escape_att_value<'a, S: Into<Cow<'a, str>>>(input: S) -> EscapedAttValue<'a> {
    let input = input.into();
    fn is_trouble(c: char) -> bool {
        c == '"' || c == '&'
    }

    if input.contains(is_trouble) {
        let mut output = String::with_capacity(input.len());
        for c in input.chars() {
            match c {
                '"' => output.push_str("&quot;"),
                '&' => output.push_str("&amp;"),
                _ => output.push(c),
            }
        }
        EscapedAttValue(Cow::Owned(output))
    } else {
        EscapedAttValue(input)
    }
}

/// XML escape an untrusted string to make it `PcdataSafe`.
pub fn escape_text<'a, S: Into<Cow<'a, str>>>(input: S) -> EscapedPcdata<'a> {
    let input = input.into();
    fn is_trouble(c: char) -> bool {
        c == '<' || c == '&'
    }

    if input.contains(is_trouble) {
        let mut output = String::with_capacity(input.len());
        for c in input.chars() {
            match c {
                '<' => output.push_str("&lt;"),
                '&' => output.push_str("&amp;"),
                _ => output.push(c),
            }
        }
        EscapedPcdata(Cow::Owned(output))
    } else {
        EscapedPcdata(input)
    }
}

#[doc(hidden)]
pub struct EscapedAttValue<'a>(pub(crate) Cow<'a, str>);
impl AttValueSafe for EscapedAttValue<'_> {}
impl Display for EscapedAttValue<'_> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.0.fmt(f)
    }
}

#[doc(hidden)]
pub struct EscapedPcdata<'a>(pub(crate) Cow<'a, str>);
impl PcdataSafe for EscapedPcdata<'_> {}
impl Display for EscapedPcdata<'_> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.0.fmt(f)
    }
}

#[cfg(test)]
mod tests {
    use crate::{escape_att_value, escape_text};

    #[test]
    fn test_escape_text() {
        assert_eq!(
            escape_text("x < 5 > 3 && \"foo\" == 'bar'").to_string(),
            "x &lt; 5 > 3 &amp;&amp; \"foo\" == 'bar'"
        );
    }

    #[test]
    fn test_escape_att_value() {
        assert_eq!(
            escape_att_value("x < 5 > 3 && \"foo\" == 'bar'").to_string(),
            "x < 5 > 3 &amp;&amp; &quot;foo&quot; == 'bar'"
        );
    }
}
