/// Wraps [`format_args`] casting all arguments to `AttValueSafe`.  The return
/// value is wrapped to also implement `AttValueSafe`.  Literal `"` and `&`
/// characters in the format string must be escaped as `&quot;` and `&amp;`.
#[macro_export]
macro_rules! format_att_value {
    ($format:expr $(,$args:expr)*) => {
        {
            // We are defining a new struct in here because otherwise it could
            // be used to circumvent the trait (we cannot just define it in
            // crate::wrappers with a private constructor since macros are
            // expanded in the calling code, where they can't call private
            // constructors defined in this crate).
            struct FormattedAttValue<'a>(std::fmt::Arguments<'a>);
            impl $crate::AttValueSafe for FormattedAttValue<'_> {}
            impl std::fmt::Display for FormattedAttValue<'_> {
                fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
                    self.0.fmt(f)
                }
            }
            FormattedAttValue(
                format_args!($format $(,&$args as &dyn $crate::AttValueSafe)*)
            )
        }
    };
}

/// Wraps [`format_args`] casting all arguments to `PcdataSafe`.  The return
/// value is wrapped to also implement `PcdataSafe`. Literal `<` and `&`
/// characters in the format string must be escaped as `&lt;` and `&amp;`
/// respectively. Do **NOT** use this to build tags (instead use the secure
/// methods from [`XmlWriter`](crate::XmlWriter)).
#[macro_export]
macro_rules! format_text {
    ($format:expr $(,$args:expr)*) => {
        {
            // See the comment in the previous macro.
            struct FormattedPcdata<'a>(std::fmt::Arguments<'a>);
            impl $crate::PcdataSafe for FormattedPcdata<'_> {}
            impl std::fmt::Display for FormattedPcdata<'_> {
                fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
                    self.0.fmt(f)
                }
            }
            FormattedPcdata(
                format_args!($format $(,&$args as &dyn $crate::PcdataSafe)*)
            )
        }
    };
}

/// Provides a minimal macro-based eDSL to structure your XML writing code.
/// See the example in the [README](crate#example).
///
/// ## Streaming attribute values
///
/// When generating attribute values you can avoid allocations by using an
/// [`AttValueWriter`](crate::AttValueWriter), for example:
///
/// ```
/// # use xmlsafe::{XmlWriter, with_writer};
/// fn generate(w: XmlWriter) -> Result<(), std::fmt::Error> {
///     with_writer!(w, {
///         tag!("example", "data" << |value| {
///             for i in 1..4 {
///                 value.write(i)?;
///                 value.write(" ")?;
///             }
///             Ok(())
///         });
///     });
///     Ok(())
/// }
///
/// let mut out = String::new();
/// generate(XmlWriter::new(&mut out)).unwrap();
/// assert_eq!(out, "<example data=\"1 2 3 \"/>");
/// ```
#[macro_export]
macro_rules! with_writer {
    // This macro is designed to play nice with rustfmt. For this reason :block
    // is used for code that should be indented and the attributes are separated
    // by commas.
    ($writer:ident, $block:block) => {
        {
            // workaround for https://github.com/rust-lang/rust/issues/35853
            macro_rules! with_dollar_sign {
                ($a:tt => $b:tt) => {
                    macro_rules! __with_dollar_sign { $a => $b }
                    __with_dollar_sign!($);
                }
            }

            let mut $writer = $writer;
            with_dollar_sign! {
                ($d:tt) => {
                    macro_rules! tag {
                        ($name:literal $d(, $k:literal$t:tt$v:expr)* $d(, $blk:block)?) => {
                            #[allow(unused_mut)]
                            let mut att_writer = $writer.open_start_tag($name)?;

                            #[allow(unused_macros)]
                            macro_rules! attr {
                                ($ak:literal << $av:expr) => {
                                    let mut att_val_writer = att_writer.key($ak)?;
                                    let fun: &dyn Fn(&mut $crate::AttValueWriter) -> Result<(), std::fmt::Error> = &$av;
                                    fun(&mut att_val_writer)?;
                                    att_writer = att_val_writer.end()?;
                                };
                                ($ak:literal=$av:expr) => {
                                    att_writer.write_attr($ak, $av)?;
                                };
                            }
                            $d(attr!($k$t$v);)*

                            macro_rules! content {
                                ($content:block) => {
                                    $writer = att_writer.close()?;
                                    $content
                                    $writer.write_end_tag($name)?;
                                };
                                () => {
                                    #[allow(unused_assignments)]
                                    {
                                        $writer = att_writer.close_empty()?;
                                    }
                                }
                            }
                            content!($d($blk)?);
                        };
                    }
                }
            }
            $block
        }
    };
}

#[cfg(test)]
mod tests {
    use std::fmt::Error;

    use crate::{AttValueWriter, XmlWriter};

    #[test]
    fn test_empty_tag() {
        fn generate(w: XmlWriter) -> Result<(), std::fmt::Error> {
            with_writer!(w, {
                tag!("example");
                tag!("example", "x" = 33);
            });
            Ok(())
        }
        let mut out = String::new();
        generate(XmlWriter::new(&mut out)).unwrap();
        assert_eq!(out, "<example/><example x=\"33\"/>");
    }

    #[test]
    fn test_stream_att_value_with_function_pointer() {
        fn gen_value(value: &mut AttValueWriter) -> Result<(), Error> {
            value.write("from function")?;
            Ok(())
        }
        fn generate(w: XmlWriter) -> Result<(), Error> {
            with_writer!(w, {
                tag!("example", "key" << gen_value);
            });
            Ok(())
        }
        let mut out = String::new();
        generate(XmlWriter::new(&mut out)).unwrap();
        assert_eq!(out, "<example key=\"from function\"/>");
    }
}
