use std::{borrow::Cow, fmt::Display};

use crate::{AttValueSafe, PcdataSafe};

pub struct EscapedAttValue<'a>(pub(crate) Cow<'a, str>);
impl AttValueSafe for EscapedAttValue<'_> {}
impl Display for EscapedAttValue<'_> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.0.fmt(f)
    }
}

pub struct EscapedPcdata<'a>(pub(crate) Cow<'a, str>);
impl PcdataSafe for EscapedPcdata<'_> {}
impl Display for EscapedPcdata<'_> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.0.fmt(f)
    }
}

pub struct FormattedAttValue<D: Display>(pub(crate) D);
impl<D: Display> AttValueSafe for FormattedAttValue<D> {}
impl<D: Display> Display for FormattedAttValue<D> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.0.fmt(f)
    }
}

pub struct FormattedPcdata<D: Display>(pub(crate) D);
impl<D: Display> PcdataSafe for FormattedPcdata<D> {}
impl<D: Display> Display for FormattedPcdata<D> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        self.0.fmt(f)
    }
}

/// Wraps [`format_args`] casting all arguments to `AttValueSafe`.  The return
/// value is wrapped to also implement `AttValueSafe`.  Literal `"` and `&`
/// characters in the format string must be escaped as `&quot;` and `&amp;`.
#[macro_export]
macro_rules! format_att_value {
    ($format:expr $(,$args:expr)*) => {
        {
            // since an exported macro cannot use a private constructor we have to use unsafe
            unsafe {
                $crate::wrappers::internal_formatted_att_value(
                    format_args!($format $(,&$args as &dyn crate::AttValueSafe)*)
                )
            }
        }
    };
}

#[doc(hidden)]
pub unsafe fn internal_formatted_att_value<D: Display>(value: D) -> FormattedAttValue<D> {
    FormattedAttValue(value)
}

/// Wraps [`format_args`] casting all arguments to `PcdataSafe`.  The return
/// value is wrapped to also implement `PcdataSafe`. Literal `<` and `&`
/// characters in the format string must be escaped as `&lt;` and `&amp;`
/// respectively. Do **NOT** use this to build tags (instead use the secure
/// methods from [`XmlWriter`](crate::XmlWriter)).
#[macro_export]
macro_rules! format_text {
    ($format:expr $(,$args:expr)*) => {
        {
            // since an exported macro cannot use a private constructor we have to use unsafe
            unsafe {
                $crate::wrappers::internal_formatted_pcdata(
                    format_args!($format $(,&$args as &dyn $crate::PcdataSafe)*)
                )
            }
        }
    };
}

#[doc(hidden)]
pub unsafe fn internal_formatted_pcdata<D: Display>(value: D) -> FormattedPcdata<D> {
    FormattedPcdata(value)
}
