// Copyright 2021 Citrix
// SPDX-License-Identifier: MIT OR Apache-2.0
// There is NO WARRANTY.

use crate::prelude::*;

#[derive(Debug)]
pub enum RefSpec<'r> {
  Def,
  Refname(&'r str),
  Tree(&'r git2::Tree<'r>),
}
impl Default for RefSpec<'_> { fn default() -> Self { RefSpec::Def } }

pub trait Repo {
  fn g2(&self) -> &git2::Repository;
  fn dir(&self) -> &str;
  fn cfg(&self) -> &Config;

  fn def_refname(&self) -> &str;

  #[throws(AE)]
  fn get_tree_maybe(&self, refname: &str) -> Option<git2::Tree> {
    let tree = (||{
      let oid = match self.g2().refname_to_id(refname) {
        Err(e) if e.code() == git2::ErrorCode::NotFound => return Ok(None),
        x => x.context("resolve ref")?,
      };
      let commit = self.g2().find_commit(oid).context("get commit")?;
      let tree = commit.tree().context("get tree")?;
      Ok::<_,AE>(Some(tree))
    })().with_context(|| refname.to_string())?;
    tree
  }

  #[throws(AE)]
  fn get_tree<'r>(&'r self, refspec: RefSpec<'r>) -> git2::Tree<'r> {
    let refname = match refspec {
      RefSpec::Def => self.def_refname(),
      RefSpec::Refname(n) => n,
      RefSpec::Tree(t) => return t.clone(),
    };
    self.get_tree_maybe(refname)?
      .ok_or_else(|| anyhow!("ref {:?} not found", refname))?
  }

  #[throws(AE)]
  fn file(&self, refspec: RefSpec, path: &str) -> Option<String> {
    let tree = self.get_tree(refspec)?;
    (||{
      let ent = match tree.get_path(path.as_ref()) {
        Err(e) if e.code() == git2::ErrorCode::NotFound => return Ok(None),
        x => x,
      }.context("get_path")?;
      let obj = ent.to_object(self.g2()).context("to_object")?;
      let blob = obj.into_blob()
        .map_err(|e| anyhow!("not a blob, but {:?}", &e.kind()))?;
      let r = str::from_utf8(blob.content()).context("from_utf8")?;
      Ok::<_,AE>(Some(r.to_owned()))
    })()
      .with_context(|| path.to_string())
      .context("look up path")?
  }

  #[throws(AE)]
  fn files(&self, refspec: RefSpec, path: &str) -> Vec<String> {
    let tree = self.get_tree(refspec)?;
    (||{
      let ent = tree.get_path(path.as_ref()).context("get_path")?;
      let obj = ent.to_object(self.g2()).context("to_object")?;
      let tree = obj.into_tree()
        .map_err(|e| anyhow!("not a tree, but {:?}", &e.kind()))?;
      tree.iter().map(|ent| Ok::<_,AE>(
        ent.name()
          .ok_or_else(|| anyhow!("bad entry in {}, not utf-8: {:?}",
                                 path, ent.name_bytes()))?
          .to_owned()          
      ))
        .collect::<Result<Vec<_>,_>>()
    })()
      .with_context(|| path.to_string())
      .context("enumerate dir")?
  }

  #[throws(AE)]
  fn run_git(&mut self, want: bool, doing: &str,
             u: &mut dyn FnMut(&mut Command) -> Result<(), AE>) {
    let mut cmd = if want {
      self.cfg().git_command()?
    } else if log_enabled!(log::Level::Debug) {
      let mut cmd = Command::new("sh");
      cmd.args(&["-c",r#"exec >&2; echo "$*";"#,"-","# git"]);
      cmd
    } else {
      let mut cmd = Command::new("true");
      cmd.arg("--");
      cmd
    };

    cmd.current_dir(self.dir());
    if let Some(v) = &self.cfg().git_ssh_command {
      cmd.env("GIT_SSH_COMMAND", v);
    }

    u(&mut cmd)?;

    let show = format!("cmd {:?}", &cmd);

    if want { debug!("{}, {}", &doing, &show); }

    let st = cmd.status().context("spawn git")?;
    if ! st.success() { throw!(
      anyhow!("git failed: {}", st)
        .context(show.clone())
        .context(self.dir().to_owned())
    ) }
  }
}

impl cache::Cache {
  #[throws(AE)]
  pub fn run_git_fetch_pull(&mut self, refresh: RefreshKind,
                            repo: &mut dyn Repo,
                            args1: &[&str],
                            remote_name: &str,
                            branch_pats: &[&str],
                            extra_refspecs: &[&str])
                            -> Option<RefreshingToken>
  {
    let want = self.cfg.want_refresh(refresh, None);

    repo.run_git(want.is_some(), "updating", &mut |cmd: &mut Command| {

      cmd.args(args1);
      if ! log_enabled!(log::Level::Debug) { cmd.arg("--quiet"); }

      cmd.arg(remote_name);
      for pat in branch_pats {
        cmd.arg(format!(
          "+refs/heads/{pat}:refs/remotes/{remote}/{pat}",
          pat=pat, remote=remote_name,
        ));
      }
      cmd.args(extra_refspecs);

      Ok::<_,AE>(())
    })
      .context("update from remote git repo")?;

    want
  }
}
