// Copyright 2021 Citrix
// SPDX-License-Identifier: MIT OR Apache-2.0
// There is NO WARRANTY.

use xensec_internal_tools::prelude::*;

#[derive(Debug,Clone,StructOpt)]
struct MainOpts {
  #[structopt(flatten)]
  config_spec: config::ConfigSpec,

  /// path to config file
  #[structopt(long, short="C", name="CONFIG.TOML")]
  config: Option<String>,

  #[structopt(flatten)]
  co: config::CommonOpts,

  #[structopt(subcommand)]  // Note that we mark a field as a subcommand
  cmd: SubOpts,
}

#[derive(Debug,Clone,StructOpt)]
enum SubOpts {
  /** populate one xsa */  Submit1 (Submit1Opts  ),
  /** test subcommand */   Test    (TestOpts     ),
  /** test one xsa */      Test1   (Test1Opts    ),
}

#[derive(Debug,Clone,StructOpt)]
struct TestOpts {
}

#[derive(Debug,Clone)]
struct AnySubcmdArgs {
  mo: MainOpts,
  cfg: Config,
}

#[throws(AE)]
fn do_test(AnySubcmdArgs{cfg,..}: AnySubcmdArgs, _so: TestOpts) {
  let ctx = Ctx::new(cfg)?;
  for x in ctx.xsas.list_by_tags()? {
    eprintln!("{:?}", &x);
  }

  let cves = analysis::Cves::ascertain(&ctx)?;
  for cve in &cves.cves {
    eprintln!("{:?}", cve);
  }
}

#[derive(Debug,Clone,StructOpt)]
struct Test1Opts {
  #[structopt(name="XSA")]
  xsa: XsaNum,
}

#[throws(AE)]
fn do_test1(AnySubcmdArgs{cfg,..}: AnySubcmdArgs, so: Test1Opts) {
  let mut ctx = Ctx::new(cfg)?;

  let tagged = ctx.xsas.list_by_tags()?;
  let mut cves = analysis::Cves::ascertain(&ctx)?;

  let got = cves.ascertain_one_xsa(&mut ctx, &tagged, so.xsa)?;
  eprintln!("{:?}", &got);
}


#[derive(Debug,Clone,StructOpt)]
struct Submit1Opts {
  #[structopt(name="XSA")]
  xsa: XsaNum,
}

#[throws(AE)]
fn do_submit1(AnySubcmdArgs{cfg,..}: AnySubcmdArgs, so: Submit1Opts) {
  let mut ctx = Ctx::new(cfg)?;
  let tagged = ctx.xsas.list_by_tags()?;
  let mut cves = analysis::Cves::ascertain(&ctx)?;
  let xstate = cves.ascertain_one_xsa(&mut ctx, &tagged, so.xsa)?;
  match &ctx.submit1(so.xsa, xstate)? {
    Right(number) => {
      println!("{} SUBMITTED {:?}", so.xsa, number);
    },
    Left(no) => {
      let mut s = String::new();
      no.report(&mut s, so.xsa).unwrap();
      print!("{}", s);
    },
  }
}

#[throws(AE)]
fn main(){
  let mut mo = MainOpts::from_args();
  mo.co.log_init()?;

  if let Some(config) = &mo.config {
    let mut cfg_data: config::ConfigSpec = (||{
      let cfg_data = fs::read_to_string(config).context("read file")?;
      let cfg_data = toml::de::from_str(&cfg_data).context("parse")?;
      Ok::<_,AE>(cfg_data)
    })().with_context(|| config.clone()).context("config file")?;

    mo.config_spec.cvelist_upstream = take(&mut cfg_data.cvelist_upstream);
    merge::Merge::merge(&mut mo.config_spec, cfg_data);
  };
  let cfg = mem::take(&mut mo.config_spec).resolve(&mo.co)?;

  let asa = AnySubcmdArgs { mo: mo.clone(), cfg };

  match mo.cmd {
    SubOpts::Submit1(so) => do_submit1(asa, so),
    SubOpts::Test   (so) => do_test   (asa, so),
    SubOpts::Test1  (so) => do_test1  (asa, so),
  }?;
}
