use reqwest::{Client, Method};
use serde::de::DeserializeOwned;
use serde::Serialize;
use serde_json::Value;

use crate::wire::{
    ApiError, BackupCreateFlags, BackupCreateRequestWire, BackupInfoWire, BackupLoadFlags,
    BackupLoadRequestWire, BackupMetaWire,
};

#[derive(Clone, Debug)]
pub struct XenonApiClient {
    host: String,
    client: Client,
}

type ApiResult<T> = Result<T, ApiError>;

impl XenonApiClient {
    pub fn new(host: String) -> Self {
        Self {
            host,
            client: Client::new(), // use builder
        }
    }

    pub async fn request<T: DeserializeOwned, D: Serialize>(
        &self,
        method: Method,
        path: &str,
        data: &D,
    ) -> ApiResult<T> {
        let url = format!("{}{}", self.host, path);

        let resp = self
            .client
            .request(method, &url)
            .json(data)
            .send()
            .await
            .map_err(|e| ApiError::Unexpected {
                message: "sending api request failed".into(),
                details: Some(e.to_string().into()),
            })?
            .json::<Value>()
            .await
            .map_err(|e| ApiError::Unexpected {
                message: "parsing api response failed".into(),
                details: Some(e.to_string().into()),
            })?;

        if let Some(Value::Bool(true)) = resp.get("success") {
            Ok(serde_json::from_value(resp.get("data").unwrap().clone()).unwrap())
        } else {
            Err(serde_json::from_value(resp).unwrap())
        }
    }

    pub async fn backup_create(
        &self,
        guild_id: u64,
        flags: BackupCreateFlags,
        message_count: u32,
    ) -> ApiResult<BackupMetaWire> {
        let path = format!("/guilds/{}/backups", guild_id);
        let data = BackupCreateRequestWire {
            message_count,
            flags: flags.bits(),
        };
        self.request(Method::POST, &path, &data).await
    }

    pub async fn backup_delete(&self, backup_id: &str) -> ApiResult<()> {
        let path = format!("/backups/{}", backup_id);
        self.request(Method::DELETE, &path, &()).await
    }

    pub async fn backup_get(&self, backup_id: &str) -> ApiResult<BackupInfoWire> {
        let path = format!("/backups/{}", backup_id);
        self.request(Method::GET, &path, &()).await
    }

    pub async fn backup_list(&self) -> ApiResult<Vec<BackupMetaWire>> {
        self.request(Method::GET, "/backups", &()).await
    }

    pub async fn backup_load(
        &self,
        guild_id: u64,
        backup_id: &str,
        flags: BackupLoadFlags,
        message_count: u32,
    ) -> ApiResult<()> {
        let path = format!("/guilds/{}/backups/{}", guild_id, backup_id);
        let data = BackupLoadRequestWire {
            message_count,
            flags: flags.bits(),
        };
        self.request(Method::POST, &path, &data).await
    }
}
