use std::borrow::Cow;
use rocket::{Request, response};
use rocket::http::Status;
use rocket::response::{content, Responder};
use serde::{Serialize, Deserialize};

#[derive(Debug, Clone, Serialize)]
pub struct ApiSuccessResponse<T: Serialize> {
    success: bool,
    data: T,
}

impl<T: Serialize> ApiSuccessResponse<T> {
    pub fn new(data: T) -> Self {
        Self {
            success: true,
            data,
        }
    }
}

impl<'r, T: Serialize> Responder<'r, 'static> for ApiSuccessResponse<T> {
    fn respond_to(self, req: &'r Request<'_>) -> response::Result<'static> {
        let string = serde_json::to_string(&self)
            .map_err(|e| {
                error!("JSON failed to serialize: {:?}", e);
                Status::InternalServerError
            })?;

        let mut resp = content::Json(string).respond_to(req)?;
        resp.set_status(Status::Ok);
        Ok(resp)
    }
}

impl<T: Serialize> From<T> for ApiSuccessResponse<T> {
    fn from(d: T) -> Self {
        Self::new(d)
    }
}

#[derive(Debug, Clone, Serialize)]
pub struct ApiErrorResponse {
    success: bool,
    #[serde(flatten)]
    error: ApiError,
}

impl ApiErrorResponse {
    pub fn new(error: ApiError) -> Self {
        Self {
            success: false,
            error,
        }
    }
}

impl<'r> Responder<'r, 'static> for ApiErrorResponse {
    fn respond_to(self, req: &'r Request<'_>) -> response::Result<'static> {
        let string = serde_json::to_string(&self)
            .map_err(|e| {
                error!("JSON failed to serialize: {:?}", e);
                Status::InternalServerError
            })?;

        let mut resp = content::Json(string).respond_to(req)?;
        resp.set_status(self.error.into());
        Ok(resp)
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
#[serde(tag = "code", rename_all = "snake_case")]
pub enum ApiError {
    NotFound {
        entity: Cow<'static, str>
    },
    MissingUserInfo,
    InvalidSnowflake,
    DatabaseFailure,
    RateLimit {
        burst_limit: u16,
        reset_after: u64,
        retry_after: f64,
    },
    Unexpected {
        message: Cow<'static, str>,
        details: Option<Cow<'static, str>>,
    },
    LimitReached {
        entity: Cow<'static, str>
    },
    InsufficientTier,
    InvalidDataField {
        field: Cow<'static, str>,
        message: Option<Cow<'static, str>>,
    },
}

impl From<ApiError> for ApiErrorResponse {
    fn from(e: ApiError) -> Self {
        Self::new(e)
    }
}

impl From<ApiError> for Status {
    fn from(e: ApiError) -> Self {
        use ApiError::*;

        match e {
            NotFound { .. } => Status::NotFound,
            MissingUserInfo => Status::Unauthorized,
            InvalidSnowflake => Status::BadRequest,
            DatabaseFailure => Status::InternalServerError,
            RateLimit { .. } => Status::TooManyRequests,
            Unexpected { .. } => Status::InternalServerError,
            LimitReached { .. } => Status::BadRequest,
            InsufficientTier => Status::BadRequest,
            InvalidDataField { .. } => Status::BadRequest
        }
    }
}

impl From<mongodb::error::Error> for ApiErrorResponse {
    fn from(e: mongodb::error::Error) -> Self {
        error!("Database error: {:?}", *e.kind);
        ApiError::DatabaseFailure.into()
    }
}

pub type ApiResponse<T> = Result<ApiSuccessResponse<T>, ApiErrorResponse>;