use serde::{Serialize, Deserialize};
use bitflags::bitflags;

bitflags! {
    pub struct BackupCreateFlags: u32 {
        const ROLES = 1 << 0;
        const CHANNELS = 1 << 1;
        const BANS = 1 << 2;
        const MEMBERS = 1 << 3;
        const MESSAGES = 1 << 4;
    }
}

impl BackupCreateFlags {
    pub fn non_premium() -> Self {
        Self::ROLES | Self::CHANNELS
    }
}

bitflags! {
    pub struct BackupLoadFlags: u32 {
        const LOAD_SETTINGS = 1 << 0;
        const DELETE_ROLES = 1 << 1;
        const DELETE_CHANNELS = 1 << 2;
        const LOAD_ROLES = 1 << 3;
        const LOAD_CHANNELS = 1 << 4;
        const LOAD_BANS = 1 << 5;
        const LOAD_MEMBERS = 1 << 6;
    }
}

impl BackupLoadFlags {
    pub fn non_premium() -> Self {
        Self::LOAD_SETTINGS | Self::DELETE_ROLES | Self::DELETE_CHANNELS | Self::LOAD_ROLES | Self::LOAD_CHANNELS
    }
}

#[derive(Debug, Serialize, Deserialize)]
pub struct BackupMetaWire {
    pub id: String,
    pub creator_id: String,
    pub timestamp: u64,
    pub guild_id: String,
    pub guild_name: String,

    pub version: u8,
    pub encrypted: bool,
    pub interval: bool,
    pub large: bool,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct BackupGuildRoleWire {
    pub id: String,
    pub name: String,
    pub position: u32,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct BackupGuildChannelWire {
    pub id: String,
    pub name: String,
    #[serde(rename = "type")]
    pub kind: u32,
    pub parent_id: Option<String>,
    pub position: u32,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct GuildStructureWire {
    pub id: String,
    pub name: String,
    pub roles: Vec<BackupGuildRoleWire>,
    pub channels: Vec<BackupGuildChannelWire>,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct BackupInfoWire {
    #[serde(flatten)]
    pub meta: BackupMetaWire,
    pub guild: GuildStructureWire,
}

#[derive(Serialize, Deserialize)]
pub struct BackupIntervalWire {
    pub last_backup: Option<u64>,
    pub next_backup: u64,
    pub interval_hours: f32,
    pub keep_count: u32,
    pub message_count: u32,
}

#[derive(Clone, Debug, Deserialize, Serialize)]
pub struct BackupCreateRequestWire {
    pub flags: u32,
    pub message_count: u32,
}

#[derive(Clone, Debug, Deserialize, Serialize)]
pub struct BackupLoadRequestWire {
    pub flags: u32,
    pub message_count: u32,
}