use serde::{Serialize, Deserialize};
use clap::Args;
use std::fs;
use std::env;
use crate::CliContext;

#[derive(Default, Debug, Clone, Args, Serialize, Deserialize)]
pub struct ConfigArgs {
    #[clap(long)]
    pub api_url: Option<String>,
    #[clap(long)]
    pub token: Option<String>,
}

pub fn get_config_path() -> String {
    let exec_file = env::current_exe().unwrap();
    let exec_dir = exec_file.parent().unwrap().to_str().unwrap();

    format!("{}/config.toml", exec_dir)
}

pub fn get_current_config() -> ConfigArgs {
    let config_path = get_config_path();
    match fs::read(&config_path) {
        Ok(b) => toml::from_slice(&b).unwrap_or_default(),
        Err(_) => ConfigArgs::default()
    }
}

pub fn save_config(config: &ConfigArgs) {
    let config_path = get_config_path();
    let bytes = toml::to_vec(config).unwrap();
    fs::write(&config_path, bytes).unwrap();
}

pub fn merge_configs(a: &mut ConfigArgs, b: ConfigArgs) {
    if b.api_url.is_some() {
        a.api_url = b.api_url;
    }

    if b.token.is_some() {
        a.token = b.token;
    }
}

pub fn handle_config(context: &CliContext, args: ConfigArgs) {
    let mut config = context.config.clone();
    merge_configs(&mut config, args);

    save_config(&config);
    println!("{:?}", config)
}