use clap::{Parser};
use xenon_lib::XenonApiClient;

pub use commands::*;

mod commands;

#[derive(Parser)]
#[clap(author, version, about, long_about = None)]
#[clap(propagate_version = true)]
struct Cli {
    #[clap(subcommand)]
    pub command: Command,
    #[clap(flatten)]
    pub config: ConfigArgs,
}

#[derive(Debug)]
pub struct CliContext {
    pub config: ConfigArgs,
    pub client: XenonApiClient,
}

#[tokio::main]
async fn main() {
    let cli = Cli::parse();

    let mut config = get_current_config();
    merge_configs(&mut config, cli.config);

    let api_url = config.api_url
        .clone()
        .unwrap_or(String::from("http://localhost:8000"));

    let context = CliContext {
        config,
        client: XenonApiClient::new(api_url),
    };

    match cli.command {
        Command::Config(args) => handle_config(&context, args),
        Command::Backup(args) => {
            match args.command {
                BackupCommand::Create(args) => handle_backup_create(&context, args).await,
                BackupCommand::Delete(args) => handle_backup_delete(&context, args).await,
                BackupCommand::Get(args) => handle_backup_get(&context, args).await,
                BackupCommand::List => handle_backup_list(&context).await,
                BackupCommand::Load(args) => handle_backup_load(&context, args).await
            }
        }
    }
}