//! ## Concurrency Safety
//! Because [Component](crate::component::Component) is just ```T : Send + Sync```.
//! [World](crate::world::World) can use [RwLock](std::sync::RwLock) to 
//! ensure the borrow check relations of all components.And [World](crate::world::World) can also
//! be ```Send + Sync```.Therefore,the all other states of world can be guarded
//! by [RwLock](std::sync::RwLock).So we can use world in concurrency environment by ```RwLock<World>```.
use crate::component::{Component, ComponentRead, ComponentStorage, ComponentWrite};
use crate::entity::{EntityBuilder, EntityId, EntityManager};
use crate::group::Group;
use crate::query::{QueryIterator, Queryable};
use crate::resource::{ResourceRead, ResourceWrite};
use crate::sparse_set::SparseSet;
use std::any::{Any, TypeId};
use std::collections::HashMap;
use std::fmt::{Debug, Formatter};
use std::sync::{RwLock, RwLockReadGuard, RwLockWriteGuard};

/// World is the core of XECS.It manages all components and entities
pub struct World {
    entity_manager: EntityManager,
    // Box<SparseSet<EntityId,Component>>
    components: HashMap<TypeId,RwLock<Box<dyn ComponentStorage>>>,
    groups: Vec<RwLock<Box<dyn Group>>>,
    resources : HashMap<TypeId,RwLock<Option<Box<dyn Any + Send + Sync>>>>
}

impl World {
    /// Create a empty world.
    pub fn new() -> World {
        World {
            entity_manager: EntityManager::new(),
            components: Default::default(),
            groups: Default::default(),
            resources : Default::default()
        }
    }

    /// Store resource in world 
    pub fn store_resource<R : 'static + Send + Sync>(&mut self,resource : R) {
        let type_id = TypeId::of::<R>();
        if !self.resources.contains_key(&type_id) {
            self.resources.insert(type_id,RwLock::new(Option::None));
        }
        self.resources.get(&type_id).unwrap() //This never fails
            .write().unwrap()
            .replace(Box::new(resource));
    }

    /// Fetch resource from world
    pub fn fetch_resource<R : 'static + Send + Sync>(&mut self) -> Option<R> {
        let type_id = TypeId::of::<R>();
        let res = self.resources.get(&type_id)?
            .write().unwrap()
            .take()?;
        // This never fails
        // because the type_id was checked before
        Some(*res.downcast::<R>().unwrap())
    }

    /// Get a read guard of resource
    pub fn resource_ref<R : 'static + Send + Sync>(&self) -> Option<ResourceRead<'_,R>> {
        let type_id = TypeId::of::<R>();
        let lock = self.resources.get(&type_id)?
            .read().unwrap();
        if lock.is_none() {
            Option::None
        } else {
            Some(ResourceRead::from_read(lock))
        }
    }

    /// Get a write guard of resource
    pub fn resource_mut<R : 'static + Send + Sync>(&self) -> Option<ResourceWrite<'_,R>> {
        let type_id = TypeId::of::<R>();
        let lock = self.resources.get(&type_id)?
            .write().unwrap();
        if lock.is_none() {
            Option::None
        } else {
            Some(ResourceWrite::from_write(lock))
        }
    }

    /// Register a component.
    /// # Panics
    /// Panic if component is registered.
    pub fn register<T: Component>(&mut self) -> &mut Self {
        assert!(!self.has_registered::<T>(),
                "World:Cannot register a component twice");
        let type_id = TypeId::of::<T>();
        self.components.insert(
            type_id,
            RwLock::new(Box::new(SparseSet::<EntityId, T>::new())),
        );
        self
    }

    /// Check if component is registered.
    pub fn has_registered<T: Component>(&self) -> bool {
        let type_id = TypeId::of::<T>();
        self.components.contains_key(&type_id)
    }

    /// Create an entity without any component in World,
    ///  return an [EntityBuilder](crate::entity::EntityBuilder).
    pub fn create_entity(&mut self) -> EntityBuilder<'_> {
        let id = self.entity_manager.create();
        EntityBuilder::new(self, id)
    }

    /// Remove entity and its components.
    pub fn remove_entity(&mut self, entity_id: EntityId) {
        assert!(self.exist(entity_id),
                "World:Cannot remove a non-exists entity");
        self.entity_manager.remove(entity_id);
        // remove entity in group
        for group in &self.groups {
            let mut group = group.write().unwrap();
            group.remove(self, entity_id);
        }
        // remove all components of this entity
        for (_, storage) in &mut self.components {
            let mut storage = storage.write().unwrap();
            if storage.has(entity_id) {
                storage.remove(entity_id);
            }
        }
    }

    /// Get lock guard of raw component storage,
    /// return None if component is not registered.
    pub(in crate) fn storage_ref(&self,id : TypeId) 
        -> Option<RwLockReadGuard<'_,Box<dyn ComponentStorage>>> {
        self.components
            .get(&id)
            .map(|rwlock|rwlock.read().unwrap())
    }

    /// Get lock guard of raw component storage,
    /// return None if component is not registered.
    pub(in crate) fn storage_mut(&self,id : TypeId) 
        -> Option<RwLockWriteGuard<'_,Box<dyn ComponentStorage>>> {
        self.components
            .get(&id)
            .map(|rwlock|rwlock.write().unwrap())
    }

    /// Attach a component to an entity.  
    /// # Panics
    /// * Panic if ```T``` is not registered.
    /// * Panic if ```entity_id``` not exist.
    pub fn attach_component<T: Component>(&mut self, entity_id: EntityId,component: T) {
        assert!(self.has_registered::<T>(),
                "World:Cannot attach component because components has not been registered.");
        assert!(self.exist(entity_id),
                "World:Cannot attach component to a non-exist entity");
        let type_id = TypeId::of::<T>();
        {
            // Unwrap never fails because assert ensures this
            let mut storage = self.storage_mut(type_id).unwrap();
            // SAFTY:
            // storage is SparseSet<EntityId,T>
            let sparse_set = unsafe {
                storage.downcast_mut::<SparseSet<EntityId,T>>()
            };
            sparse_set.add(entity_id,component);
        }
        for group in &self.groups {
            let mut group = group.write().unwrap();
            if group.type_id_a() == type_id || group.type_id_b() == type_id {
                group.add(self, entity_id);
            }
        }
    }

    /// Detach a component from an entity.
    /// # Details
    /// Return ```None``` if entity doesn't have this component,  
    /// otherwise return ```Some(component)```
    /// # Panics
    /// * Panic if ```T``` is not registered.
    /// * Panic if ```entity_id``` not exist.
    pub fn detach_component<T: Component>(&mut self, entity_id: EntityId) -> Option<T> {
        assert!(self.has_registered::<T>(),
                "World:Cannot detach component because components has not been registered.");
        assert!(self.exist(entity_id),
                "World:Cannot detach component from a non-exist entity");
        let type_id = TypeId::of::<T>();
        for group in &self.groups {
            let mut group = group.write().unwrap();
            if group.type_id_a() == type_id || group.type_id_b() == type_id {
                group.remove(self, entity_id)
            }
        }
        // Unwrap never fails because assert ensures this
        let mut storage = self.storage_mut(type_id).unwrap();
        // SAFTY:
        // storage is SparseSet<EntityId,T>
        let sparse_set = unsafe {
            storage.downcast_mut::<SparseSet<EntityId,T>>()
        };
        sparse_set.remove(entity_id)
    }

    /// Check if ```entity_id``` exists in World.
    pub fn exist(&self, entity_id: EntityId) -> bool {
        self.entity_manager.has(entity_id)
    }

    /// Get ids of all the entites.
    pub fn entities(&self) -> &[EntityId] {
        self.entity_manager.entities()
    }

    /// Get the component storage's read guard
    pub fn component_ref<T : Component>(&self) -> Option<ComponentRead<'_,T>> {
        let type_id = TypeId::of::<T>();
        let lock = self.storage_ref(type_id)?;
        Some(ComponentRead::from_lock(lock))
    }

    /// Get the component storage's write guard
    pub fn component_mut<T : Component>(&self) -> Option<ComponentWrite<'_,T>> {
        let type_id = TypeId::of::<T>();
        let lock = self.storage_mut(type_id)?;
        Some(ComponentWrite::from_lock(lock))
    }

    /// Make a [group](crate::group) to accelerate the iteration.
    /// ## Panics
    /// * Panic if ```group``` is the same as another group in [World](crate::world::World).
    /// * Panic if component is owned by another group.
    pub fn make_group<G: Group + 'static>(&mut self, group: G) {
        assert!(!self.has_group(&group),
                "World: Cannot make group because world has a same group");
        assert!(
            {
                let mut ok = true;
                'outer: for world_group in &self.groups {
                    let world_group = world_group.read().unwrap();
                    for world_group_owning in &world_group.owning_types() {
                        for owning in &group.owning_types() {
                            if *owning == *world_group_owning {
                                ok = false;
                                break 'outer;
                            }
                        }
                    }
                }
                ok
            },
            "World: Cannot make group because component was owned by another group"
        );

        let mut group = group;
        group.make_group_in_world(&self);
        self.groups.push(RwLock::new(Box::new(group)));
    }

    /// Check if (group)[crate::group] exists in [World](crate::world::World).
    /// Return true if group is same as another group in World.
    pub fn has_group<G: Group + 'static>(&self, group: &G) -> bool {
        for world_group in &self.groups {
            let world_group = world_group.read().unwrap();
            if group.type_id_a() == world_group.type_id_a()
                && group.type_id_b() == world_group.type_id_b()
                && group.owning_types() == world_group.owning_types()
            {
                return true;
            }
        }
        false
    }

    pub(in crate) fn group<G: Group + 'static>(&self, group: &G) ->RwLockReadGuard<Box<dyn Group>> {
        self.groups
            .iter()
            .find(|world_group| {
                let world_group = world_group.read().unwrap();
                group.type_id_a() == world_group.type_id_a()
                    && group.type_id_b() == world_group.type_id_b()
                    && group.owning_types() == world_group.owning_types()
            })
            // unwrap here
            // existence will be ensured by an outside function
            .unwrap()
            .read()
            .unwrap()
    }

    /// [Query](crate::query) entities with conditions
    pub fn query<'a, T: Queryable<'a>>(
        &'a self,
    ) -> Box<dyn QueryIterator<Item = <T as Queryable>::Item> + 'a> {
        <T as Queryable<'a>>::query(self)
    }

}

impl Debug for World {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        f.debug_struct("World")
            .field("entities", &self.entity_manager)
            .field(
                "components",
                &self.components.keys().cloned().collect::<Vec<TypeId>>(),
            )
            .finish()
    }
}

#[cfg(test)]
mod tests {
    use crate::component::Component;
    use std::fmt::Debug;
    use crate::group::{full_owning, non_owning, partial_owning};
    use crate::query::WithId;
    use crate::world::World;

    #[test]
    fn component_test() {
        let mut world = World::new();
        world.register::<char>();
        let id1 = world.create_entity().into_id();
        let id2 = world.create_entity().into_id();
        let _id3 = world.create_entity().into_id();

        world.attach_component(id1, 'c');
        world.attach_component(id2, 'a');

        {
            let components = world.component_ref::<char>().unwrap();
            let components = components.data();
            assert_eq!(components,&['c','a'])
        }
        world.remove_entity(id1);

        {
            let components = world.component_ref::<char>().unwrap();
            let components = components.data();
            assert_eq!(components,&['a'])
        }
    }

    #[test]
    fn group_test() {

        let mut world = World::new();

        world.register::<u32>();
        world.register::<char>();
        world.register::<()>();

        fn print<T>(world : &World,msg : &str)
        where T: Component + Clone + Debug {
            let v = world.query::<&T>()
                .with_id()
                .map(|(id,data)|(id,data.clone()))
                .collect::<Vec<_>>();
            println!("{}:{:?}",msg,&v);
        }

        world.create_entity().attach(1u32).attach(());
        let id2 = world.create_entity().attach(2u32).into_id();
        let id3 = world
            .create_entity()
            .attach(3u32)
            .attach('a')
            .attach(())
            .into_id();
        world.create_entity().attach(4u32).attach('b');
        world.create_entity().attach(5u32).attach('c');
        world.create_entity().attach(6u32);
        let id7 = world.create_entity().attach('d').attach(()).into_id();
        println!("#initial");
        print::<u32>(&world, "u32 :");
        print::<char>(&world, "char:");
        print::<()>(&world, "()  :");
        println!();

        world.make_group(full_owning::<u32, char>());
        world.make_group(non_owning::<u32, char>());
        world.make_group(partial_owning::<(), u32>());
        println!("#Made group full/non<u32,char> partial_owning<(),u32>");
        print::<u32>(&world, "u32 :");
        print::<char>(&world, "char:");
        print::<()>(&world, "()  :");
        println!();

        world.attach_component(id2,'b');
        println!("#attach component char b for id=2");
        print::<u32>(&world, "u32 :");
        print::<char>(&world, "char:");
        print::<()>(&world, "()  :");
        println!();

        world.attach_component(id7,2u32);
        println!("#attach component u32=2 for id=7");
        print::<u32>(&world, "u32 :");
        print::<char>(&world, "char:");
        print::<()>(&world, "()  :");
        println!();

        world.detach_component::<u32>(id3);
        println!("#detach component u32 for id=3");
        print::<u32>(&world, "u32 :");
        print::<char>(&world, "char:");
        print::<()>(&world, "()  :");
        println!();
    }

    #[test]
    fn debug_trait_test() {
        let mut world = World::new();

        world.register::<char>();
        world.register::<u32>();

        world.create_entity().attach('c').attach(12_u32);
        world.create_entity().attach('a');

        world.make_group(full_owning::<char, u32>());

        world.create_entity().attach('c').attach(12_u32);
        world.create_entity().attach('a');

        println!("{:?}", world);
    }

    #[test]
    fn resource_test() {
        let mut world = World::new();
        #[derive(Debug)]
        struct Test {
            name : String,
            age : u32
        }
        
        world.store_resource(Test{
            name : "affff".to_string(),
            age : 12
        });

        assert!(world.resource_ref::<Test>().is_some());
        assert_eq!(world.resource_ref::<Test>().unwrap().age,12);

        world.resource_mut::<Test>().unwrap().age = 13;

        assert_eq!(world.resource_ref::<Test>().unwrap().age,13);

        let test = world.fetch_resource::<Test>().unwrap();

        assert_eq!(test.age,13);
        assert_eq!(test.name.as_str(),"affff");

    }

}
