use std::collections::HashMap;

use crate::{config::Config, hook, portal::Interface};

#[derive(Debug)]
pub(crate) struct SessionMetadata {
    pub(crate) interface: Interface,
    pub(crate) state: SessionState,
}

#[derive(Debug)]
pub(crate) enum SessionState {
    Requested, // detected a request to CreateSession
    Created,   // detected that the response to CreateSession is successful
}

pub(crate) async fn session_created(
    session_path: String,
    cfg: &Config,
    sessions: &mut HashMap<String, SessionMetadata>,
) {
    if let Some(sm) = sessions.get(&session_path) {
        hook::session_created(&sm.interface, cfg).await;
    }
    sessions.entry(session_path).and_modify(|sm| {
        sm.state = SessionState::Created;
    });
}

pub(crate) fn session_requested(
    session_path: String,
    interface: Interface,
    sessions: &mut HashMap<String, SessionMetadata>,
) {
    sessions.insert(
        session_path,
        SessionMetadata {
            interface,
            state: SessionState::Requested,
        },
    );
}

pub(crate) async fn session_closed(
    session_path: String,
    cfg: &Config,
    sessions: &mut HashMap<String, SessionMetadata>,
) {
    if let Some(sm) = sessions.get(&session_path) {
        hook::session_closed(&sm.interface, cfg).await;
    }
    sessions.remove(&session_path);
}
