use std::io;

use tokio::process::Command;

use crate::{
    config::{CommandConfig, Config},
    portal::Interface,
};

// TODO: implement all_session_closed()

pub(crate) async fn session_closed(interface: &Interface, cfg: &Config) {
    for hook in session_closed_hooks(interface, cfg) {
        run_command(hook).await;
    }
}

pub(crate) async fn session_created(interface: &Interface, cfg: &Config) {
    for hook in session_created_hooks(interface, cfg) {
        run_command(hook).await;
    }
}

fn log_command(cmd: &Command, err: Option<io::Error>, exit_code: Option<i32>) {
    if let Some(e) = err {
        eprintln!("{:?} -> {:?}", cmd.as_std(), e);
        return;
    }
    match exit_code {
        Some(code) => eprintln!("{:?} -> exit code {}", cmd.as_std(), code),
        None => eprintln!("{:?} -> killed", cmd.as_std()),
    }
}

async fn run_command(command: CommandConfig) {
    let mut cmd = Command::new(command.cmd);
    if let Some(args) = command.args {
        for arg in args {
            cmd.arg(arg);
        }
    }
    let mut child = match cmd.spawn() {
        Ok(c) => c,
        Err(e) => {
            log_command(&cmd, Some(e), None);
            return;
        }
    };
    match child.wait().await {
        Ok(s) => log_command(&cmd, None, s.code()),
        Err(e) => {
            log_command(&cmd, Some(e), None);
            return;
        }
    }
    if let Err(e) = child.wait().await {
        log_command(&cmd, Some(e), None);
    }
}

fn session_closed_hooks(interface: &Interface, cfg: &Config) -> Vec<CommandConfig> {
    let maybe_hooks = match interface {
        Interface::Location => &cfg.location.session_closed,
        Interface::ScreenCast => &cfg.screen_cast.session_closed,
        Interface::RemoteDesktop => &cfg.remote_desktop.session_closed,
    };
    maybe_hooks.as_ref().cloned().unwrap_or_default()
}

fn session_created_hooks(interface: &Interface, cfg: &Config) -> Vec<CommandConfig> {
    let maybe_hooks = match interface {
        Interface::Location => &cfg.location.session_created,
        Interface::ScreenCast => &cfg.screen_cast.session_created,
        Interface::RemoteDesktop => &cfg.remote_desktop.session_created,
    };
    maybe_hooks.as_ref().cloned().unwrap_or_default()
}
