use super::context::Context;
use super::state::State;
use super::xutil::XUtil;
use std::error::Error;
use std::sync::Arc;
use x11rb::connection::Connection;

use x11rb::protocol::record::ConnectionExt as _;
use x11rb::protocol::xproto;

use x11rb::x11_utils::TryParse;

type Data = [u8];
fn intercept<'a, C>(
    _ctx: &Context,
    state: &State,
    data: &'a Data,
    ctrl_conn: &Arc<C>,
) -> Result<&'a Data, Box<dyn Error>>
where
    C: Connection + Send + Sync + 'static,
{
    match data[0] {
        xproto::KEY_PRESS_EVENT => {
            let (event, remaining) = xproto::KeyPressEvent::try_parse(data)?;
            debug!("KeyPress: {}", event.detail);
            let key = event.detail;

            // If the key_press_event is generated by this program,
            // ignore this event.
            if let Ok(_) = state.check_and_unmark_auto_generated_key(key) {
                debug!("Ignore auto generated key: {}", key);
                return Ok(remaining);
            }
            // if the remapped mod key is pressed,
            // mark pressed flag.
            match state.press_key(key) {
                Ok(_old) => {
                    // do nothing
                }
                Err(_reason) =>
                // if the key is NOT remapped mod key,
                // change all remapped mod keys to in-state
                {
                    state.update_all_remapped_mod_keys_to_in_used()
                }
            };
            Ok(remaining)
        }
        xproto::KEY_RELEASE_EVENT => {
            let (event, remaining) = xproto::KeyReleaseEvent::try_parse(data)?;
            debug!("KeyRelease: {}", event.detail);
            let key = event.detail;
            if let Ok(_) = state.check_and_unmark_auto_generated_key(key) {
                debug!("Ignore generated key: {}", key);
                return Ok(remaining);
            }

            if let Some(key_state) = state.remapped_key_states.borrow().get(&key) {
                if !key_state.will_be_used_as_modifier {
                    debug!("{} is not used, so generate fake key events!", key);
                    for fake_key in key_state.fake_keys.iter() {
                        // Generate KEY_PRESS_EVENT and mark flag
                        XUtil::generate_key_press_event(*fake_key, &ctrl_conn, &event)?;
                        state.mark_auto_generated_key(*fake_key);

                        // Generate KEY_RELEASE_EVENT and mark flag
                        XUtil::generate_key_release_event(*fake_key, &ctrl_conn, &event)?;
                        state.mark_auto_generated_key(*fake_key);

                        // Send pending event
                        ctrl_conn.flush()?;
                    }
                }
            }
            let _ = state.release_key(key);
            Ok(remaining)
        }
        xproto::BUTTON_PRESS_EVENT => {
            let (event, remaining) = xproto::ButtonPressEvent::try_parse(data)?;
            debug!("ButtonPress: {}", event.detail);
            state.update_all_remapped_mod_keys_to_in_used();
            state.press_mouse();
            Ok(remaining)
        }
        xproto::BUTTON_RELEASE_EVENT => {
            let (event, remaining) = xproto::ButtonReleaseEvent::try_parse(data)?;
            debug!("ButtonRelease: {}", event.detail);
            state.release_mouse();
            Ok(remaining)
        }
        _ => Ok(&data[32..]),
    }
}

pub fn run(ctx: &Context) -> Result<(), Box<dyn Error>> {
    let connections = XUtil::create_connections()?;
    let ctrl_conn = Arc::new(connections.0);
    let data_conn = Arc::new(connections.1);

    let record_context = ctrl_conn.generate_id()?;
    XUtil::create_record_context(ctx, Arc::clone(&ctrl_conn), record_context)?;
    const START_OF_DATA: u8 = 4;
    const RECORD_FROM_SERVER: u8 = 0;
    let state = State::new(&ctx);
    for reply in data_conn.record_enable_context(record_context)? {
        let reply = reply?;
        if reply.client_swapped {
            warn!("Byte swapped clients are unsupported");
        } else if reply.category == RECORD_FROM_SERVER {
            let mut remaining = &reply.data[..];
            while !remaining.is_empty() {
                remaining = intercept(&ctx, &state, &reply.data, &ctrl_conn)?;
            }
        } else if reply.category == START_OF_DATA {
            debug!("Start Of Date");
        } else {
            warn!("Got a reply with an unsupported category: {:?}", reply);
        }
    }

    println!("main logic here {:?}", ctx.is_debug_mode());
    Ok(())
}
