use log::{debug, trace, warn};
use tokio::io::unix::AsyncFd;
use tokio::time::Instant;
use x11rb::connection::Connection as _;
use x11rb::connection::RequestConnection as _;
use x11rb::cookie::Cookie;
use x11rb::protocol::xfixes::{self, ConnectionExt as _};
use x11rb::protocol::xproto::{self, ConnectionExt as _, CursorWrapper, WindowWrapper};
use x11rb::protocol::Event;
use zeroize::Zeroize;

use crate::backbuffer::Backbuffer;
use crate::dialog::{Action, Dialog};
use crate::errors::{Error, Result, Unsupported};
use crate::keyboard::Keyboard;
use crate::secret::Passphrase;
use crate::Connection;

enum State {
    Continue,
    Ready,
    Cancelled,
}

#[derive(Debug)]
enum Reply {
    GrabKeyboard(xproto::GrabKeyboardReply),
    Selection(xproto::GetPropertyReply),
}

#[allow(clippy::struct_excessive_bools)]
pub struct XContext<'a> {
    pub xfd: &'a AsyncFd<Connection>,
    pub backbuffer: Backbuffer<'a>,
    pub(super) window: WindowWrapper<'a, Connection>,
    pub keyboard: Keyboard<'a>,
    pub(super) atoms: crate::AtomCollection,
    pub(super) width: u16,
    pub(super) height: u16,
    pub(super) grab_keyboard: bool,
    pub(super) startup_time: Instant,
    pub(super) keyboard_grabbed: bool,
    pub(super) input_cursor: Option<CursorWrapper<'a, Connection>>,
    pub(super) compositor_atom: Option<xproto::Atom>,
    pub(super) grab_keyboard_cookie: Option<Cookie<'a, Connection, xproto::GrabKeyboardReply>>,
    pub(super) selection_cookie: Option<Cookie<'a, Connection, xproto::GetPropertyReply>>,
    pub(super) debug: bool,
    pub(super) first_expose_received: bool,
}

impl<'a> XContext<'a> {
    pub fn conn(&self) -> &'a Connection {
        self.xfd.get_ref()
    }

    pub fn init(&self) -> Result<()> {
        if let Some(compositor_atom) = self.compositor_atom {
            self.conn()
                .extension_information(xfixes::X11_EXTENSION_NAME)?
                .ok_or_else(|| Unsupported("x11 xfixes extension required".into()))?;
            let (major, minor) = xfixes::X11_XML_VERSION;
            let version_cookie = self.conn().xfixes_query_version(major, minor)?;
            if log::log_enabled!(log::Level::Debug) {
                let version = version_cookie.reply()?;
                debug!(
                    "xfixes version {}.{}",
                    version.major_version, version.minor_version
                );
            }

            self.conn().xfixes_select_selection_input(
                self.window.window(),
                compositor_atom,
                xfixes::SelectionEventMask::SET_SELECTION_OWNER
                    | xfixes::SelectionEventMask::SELECTION_WINDOW_DESTROY
                    | xfixes::SelectionEventMask::SELECTION_CLIENT_CLOSE,
            )?;
        }
        Ok(())
    }

    // TODO want cookie.poll_for_reply
    fn poll_for_reply(&mut self) -> Result<Option<Reply>> {
        Ok(if let Some(cookie) = self.selection_cookie.take() {
            Some(Reply::Selection(cookie.reply()?))
        } else if let Some(cookie) = self.grab_keyboard_cookie.take() {
            Some(Reply::GrabKeyboard(cookie.reply()?))
        } else {
            None
        })
    }

    pub async fn run_events(&mut self, mut dialog: Dialog) -> Result<Option<Passphrase>> {
        tokio::pin! { let events_ready = self.xfd.readable(); }
        dialog.init_events();
        loop {
            // TODO do not draw if the window is not exposed at all
            self.backbuffer.commit(&mut dialog)?;
            self.conn().flush()?;
            tokio::select! {
                action = dialog.handle_events() => {
                    if matches!(action, Action::Cancel) {
                        return Ok(None)
                    }
                }
                events_guard = &mut events_ready => {
                    let state = if let Some(event) = self.conn().poll_for_event()? {
                        if self.debug {
                            trace!("event {:?}", event);
                        }
                        self.handle_event(&mut dialog, event)?
                    } else if let Some(reply) = self.poll_for_reply()? {
                        if self.debug {
                            trace!("reply {:?}", reply);
                        }
                        self.handle_reply(&mut dialog, reply);
                        State::Continue
                    } else {
                        events_guard.unwrap().clear_ready();
                        State::Continue
                    };
                    events_ready.set(self.xfd.readable());
                    match state {
                        State::Continue => {},
                        State::Ready => { return Ok(Some(dialog.indicator.into_pass())) },
                        State::Cancelled => { return Ok(None) },
                    }
                }
            }
            tokio::task::yield_now().await;
        }
    }

    pub fn set_default_cursor(&self) -> Result<()> {
        self.conn().change_window_attributes(
            self.window.window(),
            &xproto::ChangeWindowAttributesAux::new().cursor(x11rb::NONE),
        )?;
        Ok(())
    }

    pub fn set_input_cursor(&self) -> Result<()> {
        trace!("set input cursor");
        if let Some(ref cursor) = self.input_cursor {
            self.conn().change_window_attributes(
                self.window.window(),
                &xproto::ChangeWindowAttributesAux::new().cursor(cursor.cursor()),
            )?;
            trace!("input cursor set");
        }
        Ok(())
    }

    pub fn paste_primary(&self) -> Result<()> {
        trace!("PRIMARY selection");
        self.conn().convert_selection(
            self.window.window(),
            xproto::AtomEnum::PRIMARY.into(),
            self.atoms.UTF8_STRING,
            self.atoms.XSEL_DATA,
            x11rb::CURRENT_TIME,
        )?;
        Ok(())
    }

    pub fn paste_clipboard(&self) -> Result<()> {
        self.conn().convert_selection(
            self.window.window(),
            self.atoms.CLIPBOARD,
            self.atoms.UTF8_STRING,
            self.atoms.XSEL_DATA,
            x11rb::CURRENT_TIME,
        )?;
        Ok(())
    }

    #[allow(clippy::too_many_lines)]
    fn handle_event(&mut self, dialog: &mut Dialog, event: Event) -> Result<State> {
        match event {
            Event::Error(error) => {
                return Err(Error::X11(error));
            }
            Event::Expose(expose_event) => {
                if expose_event.count > 0 {
                    return Ok(State::Continue);
                }

                self.backbuffer.set_exposed();

                if !self.first_expose_received {
                    debug!(
                        "time until first expose {}ms",
                        self.startup_time.elapsed().as_millis()
                    );
                    self.first_expose_received = true;
                }

                if self.grab_keyboard && !self.keyboard_grabbed {
                    if self.grab_keyboard_cookie.is_some() {
                        debug!("grab keyboard already requested");
                    } else {
                        self.grab_keyboard_cookie = Some(self.conn().grab_keyboard(
                            false,
                            self.window.window(),
                            x11rb::CURRENT_TIME,
                            xproto::GrabMode::ASYNC,
                            xproto::GrabMode::ASYNC,
                        )?);
                    }
                }
            }
            Event::ConfigureNotify(ev) => {
                if self.width != ev.width || self.height != ev.height {
                    trace!("resize event w: {}, h: {}", ev.width, ev.height);
                    self.width = ev.width;
                    self.height = ev.height;
                    self.backbuffer.resize_requested = Some((ev.width, ev.height));
                }
            }
            Event::MotionNotify(me) => {
                if me.same_screen {
                    let (x, y) = self
                        .backbuffer
                        .cr
                        .device_to_user(f64::from(me.event_x), f64::from(me.event_y))
                        .expect("cairo device_to_user");
                    dialog.handle_motion(x, y, self)?;
                } else {
                    trace!("not same screen");
                }
            }
            // both events have the same structure
            Event::ButtonPress(bp) | Event::ButtonRelease(bp) => {
                let isrelease = matches!(event, Event::ButtonRelease(_));
                trace!(
                    "button {}: {:?}",
                    if isrelease { "release" } else { "press" },
                    bp
                );
                if !bp.same_screen {
                    trace!("not same screen");
                    return Ok(State::Continue);
                }
                let (x, y) = self
                    .backbuffer
                    .cr
                    .device_to_user(f64::from(bp.event_x), f64::from(bp.event_y))
                    .expect("cairo device_to_user");
                let action = dialog.handle_button_press(bp.detail.into(), x, y, isrelease, self)?;
                match action {
                    Action::Ok => return Ok(State::Ready),
                    Action::Cancel => return Ok(State::Cancelled),
                    Action::Nothing => {}
                    _ => unreachable!(),
                }
            }
            Event::KeyPress(key_press) => {
                let action = dialog.handle_key_press(key_press.detail.into(), self)?;
                trace!("action {:?}", action);
                match action {
                    Action::Ok => return Ok(State::Ready),
                    Action::Cancel => return Ok(State::Cancelled),
                    Action::Nothing => {}
                    _ => unreachable!(),
                }
            }
            Event::SelectionNotify(sn) => {
                if sn.property == x11rb::NONE {
                    warn!("invalid selection");
                    return Ok(State::Continue);
                }
                self.selection_cookie = Some(self.conn().get_property(
                    false,
                    sn.requestor,
                    sn.property,
                    xproto::GetPropertyType::ANY,
                    0,
                    u32::MAX,
                )?);
            }
            Event::FocusIn(fe) => {
                if fe.mode == xproto::NotifyMode::GRAB {
                    self.keyboard_grabbed = true;
                } else if fe.mode == xproto::NotifyMode::UNGRAB {
                    self.keyboard_grabbed = false;
                }
                dialog.indicator.set_focused(true);
            }
            Event::FocusOut(fe) => {
                if fe.mode == xproto::NotifyMode::GRAB {
                    self.keyboard_grabbed = true;
                } else if fe.mode == xproto::NotifyMode::UNGRAB {
                    self.keyboard_grabbed = false;
                }
                if fe.mode != xproto::NotifyMode::GRAB
                    && fe.mode != xproto::NotifyMode::WHILE_GRABBED
                {
                    dialog.indicator.set_focused(false);
                }
            }
            Event::ClientMessage(client_message) => {
                debug!("client message");
                if client_message.format == 32
                    && client_message.data.as_data32()[0] == self.atoms.WM_DELETE_WINDOW
                {
                    debug!("close requested");
                    return Ok(State::Cancelled);
                }
            }
            Event::PresentIdleNotify(ev) => {
                self.backbuffer.on_idle_notify(&ev);
            }
            Event::PresentCompleteNotify(ev) => {
                self.backbuffer.on_vsync_completed(ev);
            }
            Event::XkbStateNotify(key) => {
                self.keyboard.update_mask(&key);
            }
            // TODO needs more testing
            Event::XkbNewKeyboardNotify(..) => {
                debug!("xkb new keyboard notify");
                self.keyboard.reload_keymap();
            }
            // TODO needs more testing
            Event::XkbMapNotify(..) => {
                debug!("xkb map notify");
                self.keyboard.reload_keymap();
            }
            Event::XfixesSelectionNotify(sn) => {
                debug!("selection notify: {:?}", sn);
                dialog.set_transparency(sn.subtype == xfixes::SelectionEvent::SET_SELECTION_OWNER);
            }
            // minimized
            Event::UnmapNotify(..) => {
                debug!("set invisible");
                self.backbuffer.visible = false;
            }
            // Ignored events:
            // unminimized
            Event::MapNotify(..) | Event::ReparentNotify(..) | Event::KeyRelease(..) => {
                trace!("ignored event {:?}", event);
            }
            event => {
                debug!("unexpected event {:?}", event);
            }
        }
        Ok(State::Continue)
    }

    fn handle_reply(&mut self, dialog: &mut Dialog, reply: Reply) {
        match reply {
            Reply::GrabKeyboard(gk) => {
                let grabbed = gk.status;
                match grabbed {
                    xproto::GrabStatus::SUCCESS => debug!("keyboard grab succeeded"),
                    xproto::GrabStatus::ALREADY_GRABBED => debug!("keyboard already grabbed"),
                    _ => warn!("keyboard grab failed: {:?}", grabbed),
                }
            }
            Reply::Selection(selection) => {
                if selection.format != 8 {
                    warn!("invalid selection format {}", selection.format);
                    return;
                // TODO
                } else if selection.type_ == self.atoms.INCR {
                    warn!("Selection too big and INCR selection not implemented");
                    return;
                }
                match String::from_utf8(selection.value) {
                    Err(err) => {
                        warn!("selection is not valid utf8: {}", err);
                        err.into_bytes().zeroize();
                    }
                    Ok(mut val) => {
                        dialog.indicator.pass_insert(&val, true);
                        val.zeroize();
                    }
                }
            }
        }
    }
}

impl<'a> Drop for XContext<'a> {
    fn drop(&mut self) {
        if self.keyboard_grabbed {
            if let Err(err) = self.conn().ungrab_keyboard(x11rb::CURRENT_TIME) {
                debug!("ungrab keyboard failed: {}", err);
            }
        }
        if let Some(compositor_atom) = self.compositor_atom {
            if let Err(err) = xfixes::select_selection_input(
                self.conn(),
                self.window.window(),
                compositor_atom,
                0_u32,
            ) {
                debug!("clear select selection failed: {}", err);
            }
        }
        debug!("dropping XContext");
        if let Err(err) = self.conn().flush() {
            debug!("conn flush failed: {}", err);
        }
    }
}
