use const_oid::db::rfc6960::ID_PKIX_OCSP_BASIC;
use der::asn1::{Null, ObjectIdentifier};
use der::{Decode, Encode};
use hex_literal::hex;
use x509_cert::ext::pkix::CrlReason;
use x509_ocsp::Version::V1;
use x509_ocsp::*;

#[test]
fn decode_ocsp_req_ca_signed() {
    // request generated using openssl via this command:
    // openssl ocsp -noverify -no_nonce -respout ~/Desktop/ocspdigicert.resp
    // -reqout ~/Desktop/ocspdigicert.req -issuer ~/Desktop/DigiCertGlobalCAG2.crt
    // -cert ~/Desktop/amazon.der -url http://ocsp.digicert.com -header "HOST" "ocsp.digicert.com" -text

    pub const PKIXALG_SHA1: ObjectIdentifier = ObjectIdentifier::new_unwrap("1.3.14.3.2.26");

    let ocsp_req =
        OcspRequest::from_der(&hex!("3051304F304D304B3049300906052B0E03021A05000414A87E303106E4E88565CFE952598FA6DA7C00532F0414246E2B2DD06A925151256901AA9A47A689E7402002100E4239AB85E2E6A27C52C6DE9B9078D9")[..]).unwrap();
    assert_eq!(ocsp_req.tbs_request.version, V1);
    //assert!(ocsp_req.tbs_request.requestor_name.is_none());
    assert_eq!(ocsp_req.tbs_request.request_list.len(), 1);
    let req = &ocsp_req.tbs_request.request_list[0];
    assert_eq!(req.req_cert.hash_algorithm.oid, PKIXALG_SHA1);
    assert!(req.req_cert.hash_algorithm.parameters.is_some());
    assert_eq!(
        req.req_cert
            .hash_algorithm
            .parameters
            .to_vec()
            .unwrap()
            .as_slice(),
        Null.to_vec().unwrap().as_slice()
    );
    assert_eq!(
        req.req_cert.issuer_name_hash.as_bytes(),
        &hex!("A87E303106E4E88565CFE952598FA6DA7C00532F")
    );
    assert_eq!(
        req.req_cert.issuer_key_hash.as_bytes(),
        &hex!("246E2B2DD06A925151256901AA9A47A689E74020")
    );
    assert_eq!(
        req.req_cert.serial_number.as_bytes(),
        &hex!("0E4239AB85E2E6A27C52C6DE9B9078D9")
    );

    let reenc = ocsp_req.to_vec().unwrap();
    assert_eq!(reenc, &hex!("3051304F304D304B3049300906052B0E03021A05000414A87E303106E4E88565CFE952598FA6DA7C00532F0414246E2B2DD06A925151256901AA9A47A689E7402002100E4239AB85E2E6A27C52C6DE9B9078D9")[..])
}

#[test]
fn decode_ocsp_resp_ca_signed() {
    // response generated using openssl via this command:
    // openssl ocsp -noverify -no_nonce -respout ~/Desktop/ocspdigicert.resp
    // -reqout ~/Desktop/ocspdigicert.req -issuer ~/Desktop/DigiCertGlobalCAG2.crt
    // -cert ~/Desktop/amazon.der -url http://ocsp.digicert.com -header "HOST" "ocsp.digicert.com" -text

    // 308201D30A0100A08201CC308201C806092B0601050507300101048201B9308201B530819EA2160414246E2B2DD06A925151256901AA9A47A689E74020180F32303232303230323034313232355A307330713049300906052B0E03021A05000414A87E303106E4E88565CFE952598FA6DA7C00532F0414246E2B2DD06A925151256901AA9A47A689E7402002100E4239AB85E2E6A27C52C6DE9B9078D98000180F32303232303230323033353730315AA011180F32303232303230393033313230315A300D06092A864886F70D01010B0500038201010089DC0913BD91DB172C314AE8F78E6574950353618FFBBF2F12B1F134F21EC20063E66C60AC463BDB926A9D5B699EB39689032659D43C441863BAC9D8B7CC84137B558503D2631348EDA79AEC9FD4B69AE988F68A752470D3D4BA46F41B9FD41A974A5311BAA191AFBA370F0B051F035297FA76A2133B98FF471BDA4BFBFDA275B4FA1CF8377EEB2B2141F9C998ACE27C60F11E221D6732A5D538FEDF9A01D776E1877BCF70A8C953630B2C66752A66DF087C075794F5A2C0023C964CD73476EA2556F091F6C191B30B74E2523F668F32CE10AFB3D51CE053D4D41E62214B60171093AB915DBBA497F09B3C279127E1D25C7AA6A7048BBEA41A23F4F827BD108F
    let ocsp_resp =
        OcspResponse::from_der(&hex!("308201D30A0100A08201CC308201C806092B0601050507300101048201B9308201B530819EA2160414246E2B2DD06A925151256901AA9A47A689E74020180F32303232303230323034313232355A307330713049300906052B0E03021A05000414A87E303106E4E88565CFE952598FA6DA7C00532F0414246E2B2DD06A925151256901AA9A47A689E7402002100E4239AB85E2E6A27C52C6DE9B9078D98000180F32303232303230323033353730315AA011180F32303232303230393033313230315A300D06092A864886F70D01010B0500038201010089DC0913BD91DB172C314AE8F78E6574950353618FFBBF2F12B1F134F21EC20063E66C60AC463BDB926A9D5B699EB39689032659D43C441863BAC9D8B7CC84137B558503D2631348EDA79AEC9FD4B69AE988F68A752470D3D4BA46F41B9FD41A974A5311BAA191AFBA370F0B051F035297FA76A2133B98FF471BDA4BFBFDA275B4FA1CF8377EEB2B2141F9C998ACE27C60F11E221D6732A5D538FEDF9A01D776E1877BCF70A8C953630B2C66752A66DF087C075794F5A2C0023C964CD73476EA2556F091F6C191B30B74E2523F668F32CE10AFB3D51CE053D4D41E62214B60171093AB915DBBA497F09B3C279127E1D25C7AA6A7048BBEA41A23F4F827BD108F")[..]).unwrap();

    assert_eq!(ocsp_resp.response_status, OcspResponseStatus::Successful);
    assert!(ocsp_resp.response_bytes.is_some());
    let response_bytes = ocsp_resp.response_bytes.as_ref().unwrap();
    assert_eq!(response_bytes.response_type, ID_PKIX_OCSP_BASIC);

    pub const PKIXALG_SHA256_WITH_RSA_ENCRYPTION: ObjectIdentifier =
        ObjectIdentifier::new_unwrap("1.2.840.113549.1.1.11");
    pub const PKIXALG_SHA1: ObjectIdentifier = ObjectIdentifier::new_unwrap("1.3.14.3.2.26");

    let bor = BasicOcspResponse::from_der(response_bytes.response.as_bytes()).unwrap();
    assert_eq!(
        bor.signature_algorithm.oid,
        PKIXALG_SHA256_WITH_RSA_ENCRYPTION
    );
    assert!(bor.signature_algorithm.parameters.is_some());
    assert_eq!(
        bor.signature_algorithm
            .parameters
            .to_vec()
            .unwrap()
            .as_slice(),
        Null.to_vec().unwrap().as_slice()
    );

    assert_eq!(bor.tbs_response_data.version, V1);

    match bor.tbs_response_data.responder_id {
        ResponderId::ByKey(kh) => {
            assert_eq!(
                kh.as_bytes(),
                &hex!("246E2B2DD06A925151256901AA9A47A689E74020")
            );
        }
        _ => {
            panic!("Expected ByKey and got something else")
        }
    }

    assert_eq!(
        bor.tbs_response_data
            .produced_at
            .to_unix_duration()
            .as_secs(),
        1643775145
    );
    assert_eq!(bor.tbs_response_data.responses.len(), 1);
    let sr = &bor.tbs_response_data.responses[0];
    assert_eq!(sr.cert_id.hash_algorithm.oid, PKIXALG_SHA1);
    assert!(sr.cert_id.hash_algorithm.parameters.is_some());
    assert_eq!(
        sr.cert_id
            .hash_algorithm
            .parameters
            .to_vec()
            .unwrap()
            .as_slice(),
        Null.to_vec().unwrap().as_slice()
    );
    assert_eq!(
        sr.cert_id.issuer_name_hash.as_bytes(),
        &hex!("A87E303106E4E88565CFE952598FA6DA7C00532F")
    );
    assert_eq!(
        sr.cert_id.issuer_key_hash.as_bytes(),
        &hex!("246E2B2DD06A925151256901AA9A47A689E74020")
    );
    assert_eq!(
        sr.cert_id.serial_number.as_bytes(),
        &hex!("0E4239AB85E2E6A27C52C6DE9B9078D9")
    );

    match sr.cert_status {
        CertStatus::Good(g) => {
            assert_eq!(
                g.to_vec().unwrap().as_slice(),
                Null.to_vec().unwrap().as_slice()
            );
        }
        _ => {
            panic!("Expected Good and got something other")
        }
    }

    assert_eq!(sr.this_update.to_unix_duration().as_secs(), 1643774221);
    assert!(sr.next_update.is_some());
    assert_eq!(
        sr.next_update.unwrap().to_unix_duration().as_secs(),
        1644376321
    );

    let reenc = ocsp_resp.to_vec().unwrap();
    assert_eq!(reenc, &hex!("308201D30A0100A08201CC308201C806092B0601050507300101048201B9308201B530819EA2160414246E2B2DD06A925151256901AA9A47A689E74020180F32303232303230323034313232355A307330713049300906052B0E03021A05000414A87E303106E4E88565CFE952598FA6DA7C00532F0414246E2B2DD06A925151256901AA9A47A689E7402002100E4239AB85E2E6A27C52C6DE9B9078D98000180F32303232303230323033353730315AA011180F32303232303230393033313230315A300D06092A864886F70D01010B0500038201010089DC0913BD91DB172C314AE8F78E6574950353618FFBBF2F12B1F134F21EC20063E66C60AC463BDB926A9D5B699EB39689032659D43C441863BAC9D8B7CC84137B558503D2631348EDA79AEC9FD4B69AE988F68A752470D3D4BA46F41B9FD41A974A5311BAA191AFBA370F0B051F035297FA76A2133B98FF471BDA4BFBFDA275B4FA1CF8377EEB2B2141F9C998ACE27C60F11E221D6732A5D538FEDF9A01D776E1877BCF70A8C953630B2C66752A66DF087C075794F5A2C0023C964CD73476EA2556F091F6C191B30B74E2523F668F32CE10AFB3D51CE053D4D41E62214B60171093AB915DBBA497F09B3C279127E1D25C7AA6A7048BBEA41A23F4F827BD108F")[..])
}

#[test]
fn decode_ocsp_req_delegated() {
    // request generated using openssl via this command:
    // openssl ocsp -noverify -no_nonce -respout ~/Desktop/ocspdod.resp -reqout ~/Desktop/ocspdod.req
    // -issuer ~/Desktop/email_ca_59.der -cert ~/Desktop/ee.der -url http://ocsp.disa.mil
    // -header "HOST" "ocsp.disa.mil" -text

    pub const PKIXALG_SHA1: ObjectIdentifier = ObjectIdentifier::new_unwrap("1.3.14.3.2.26");

    let ocsp_req =
        OcspRequest::from_der(&hex!("304530433041303F303D300906052B0E03021A050004140F0D5890F551D42ACF5431B7F42A321F7B74A4730414771441A65D9526D01DFF953B628CEAB7B55D3B92020401017467")[..]).unwrap();
    assert_eq!(ocsp_req.tbs_request.version, V1);
    //assert!(ocsp_req.tbs_request.requestor_name.is_none());
    assert_eq!(ocsp_req.tbs_request.request_list.len(), 1);
    let req = &ocsp_req.tbs_request.request_list[0];
    assert_eq!(req.req_cert.hash_algorithm.oid, PKIXALG_SHA1);
    assert!(req.req_cert.hash_algorithm.parameters.is_some());
    assert_eq!(
        req.req_cert
            .hash_algorithm
            .parameters
            .to_vec()
            .unwrap()
            .as_slice(),
        Null.to_vec().unwrap().as_slice()
    );
    assert_eq!(
        req.req_cert.issuer_name_hash.as_bytes(),
        &hex!("0F0D5890F551D42ACF5431B7F42A321F7B74A473")
    );
    assert_eq!(
        req.req_cert.issuer_key_hash.as_bytes(),
        &hex!("771441A65D9526D01DFF953B628CEAB7B55D3B92")
    );
    assert_eq!(req.req_cert.serial_number.as_bytes(), &hex!("01017467"));

    let reenc = ocsp_req.to_vec().unwrap();
    assert_eq!(reenc, &hex!("304530433041303F303D300906052B0E03021A050004140F0D5890F551D42ACF5431B7F42A321F7B74A4730414771441A65D9526D01DFF953B628CEAB7B55D3B92020401017467")[..])
}

#[test]
fn decode_ocsp_resp_delegated() {
    // response generated using openssl via this command:
    // openssl ocsp -noverify -no_nonce -respout ~/Desktop/ocspdod.resp -reqout ~/Desktop/ocspdod.req
    // -issuer ~/Desktop/email_ca_59.der -cert ~/Desktop/ee.der -url http://ocsp.disa.mil
    // -header "HOST" "ocsp.disa.mil" -text

    // 308201D30A0100A08201CC308201C806092B0601050507300101048201B9308201B530819EA2160414246E2B2DD06A925151256901AA9A47A689E74020180F32303232303230323034313232355A307330713049300906052B0E03021A05000414A87E303106E4E88565CFE952598FA6DA7C00532F0414246E2B2DD06A925151256901AA9A47A689E7402002100E4239AB85E2E6A27C52C6DE9B9078D98000180F32303232303230323033353730315AA011180F32303232303230393033313230315A300D06092A864886F70D01010B0500038201010089DC0913BD91DB172C314AE8F78E6574950353618FFBBF2F12B1F134F21EC20063E66C60AC463BDB926A9D5B699EB39689032659D43C441863BAC9D8B7CC84137B558503D2631348EDA79AEC9FD4B69AE988F68A752470D3D4BA46F41B9FD41A974A5311BAA191AFBA370F0B051F035297FA76A2133B98FF471BDA4BFBFDA275B4FA1CF8377EEB2B2141F9C998ACE27C60F11E221D6732A5D538FEDF9A01D776E1877BCF70A8C953630B2C66752A66DF087C075794F5A2C0023C964CD73476EA2556F091F6C191B30B74E2523F668F32CE10AFB3D51CE053D4D41E62214B60171093AB915DBBA497F09B3C279127E1D25C7AA6A7048BBEA41A23F4F827BD108F
    let ocsp_resp =
        OcspResponse::from_der(&hex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unwrap();

    assert_eq!(ocsp_resp.response_status, OcspResponseStatus::Successful);
    assert!(ocsp_resp.response_bytes.is_some());
    let response_bytes = ocsp_resp.response_bytes.as_ref().unwrap();
    assert_eq!(response_bytes.response_type, ID_PKIX_OCSP_BASIC);

    pub const PKIXALG_SHA256_WITH_RSA_ENCRYPTION: ObjectIdentifier =
        ObjectIdentifier::new_unwrap("1.2.840.113549.1.1.11");
    pub const PKIXALG_SHA1: ObjectIdentifier = ObjectIdentifier::new_unwrap("1.3.14.3.2.26");

    let bor = BasicOcspResponse::from_der(response_bytes.response.as_bytes()).unwrap();
    assert_eq!(
        bor.signature_algorithm.oid,
        PKIXALG_SHA256_WITH_RSA_ENCRYPTION
    );
    assert!(bor.signature_algorithm.parameters.is_some());
    assert_eq!(
        bor.signature_algorithm
            .parameters
            .to_vec()
            .unwrap()
            .as_slice(),
        Null.to_vec().unwrap().as_slice()
    );

    assert_eq!(bor.tbs_response_data.version, V1);

    match bor.tbs_response_data.responder_id {
        ResponderId::ByKey(kh) => {
            assert_eq!(
                kh.as_bytes(),
                &hex!("ADB0A9B2DDE9D444B4DF80F599598E84AC5EC687")
            );
        }
        _ => {
            panic!("Expected ByKey and got something else")
        }
    }

    assert_eq!(
        bor.tbs_response_data
            .produced_at
            .to_unix_duration()
            .as_secs(),
        1643900556
    );
    assert_eq!(bor.tbs_response_data.responses.len(), 20);
    let sr = &bor.tbs_response_data.responses[10];

    assert_eq!(sr.cert_id.hash_algorithm.oid, PKIXALG_SHA1);
    assert!(sr.cert_id.hash_algorithm.parameters.is_some());
    assert_eq!(
        sr.cert_id
            .hash_algorithm
            .parameters
            .to_vec()
            .unwrap()
            .as_slice(),
        Null.to_vec().unwrap().as_slice()
    );
    assert_eq!(
        sr.cert_id.issuer_name_hash.as_bytes(),
        &hex!("0F0D5890F551D42ACF5431B7F42A321F7B74A473")
    );
    assert_eq!(
        sr.cert_id.issuer_key_hash.as_bytes(),
        &hex!("771441A65D9526D01DFF953B628CEAB7B55D3B92")
    );
    assert_eq!(sr.cert_id.serial_number.as_bytes(), &hex!("01017467"));

    match &sr.cert_status {
        CertStatus::Revoked(ri) => {
            assert!(ri.revocation_reason.is_some());
            assert_eq!(ri.revocation_reason.unwrap(), CrlReason::AffiliationChanged,);
            assert_eq!(ri.revocation_time.to_unix_duration().as_secs(), 1632934667,);
        }
        _ => {
            panic!("Expected Good and got something other")
        }
    }

    assert_eq!(sr.this_update.to_unix_duration().as_secs(), 1643848200);
    assert!(sr.next_update.is_some());
    assert_eq!(
        sr.next_update.unwrap().to_unix_duration().as_secs(),
        1644456600
    );

    let reenc = ocsp_resp.to_vec().unwrap();
    assert_eq!(reenc, &hex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
}
