use crate::ascii::ETX;
use crate::{Address, Parameter, Value};
use std::convert::TryFrom;

pub enum ParseError {
    Error,
    NeedData(usize),
}

type ParseResult<'buf, OK> = Result<(&'buf [u8], OK), ParseError>;

pub mod node {
    pub use crate::nom_parser::node::CommandToken;

    pub fn parse_command(buf: &[u8]) -> (usize, CommandToken) {}
}

fn address(buf: &[u8]) -> ParseResult<Address> {
    if buf.len() < 4 {
        return Err(ParseError::NeedData(4 - buf.len()));
    }
    if buf[0] != buf[1] || buf[2] != buf[3] || !buf[1].is_ascii_digit() || !buf[2].is_ascii_digit()
    {
        return Err(ParseError::Error);
    }

    Address::new((buf[1] - b'0') * 10 + buf[2] - b'0')
        .map(|addr| (&buf[4..], addr))
        .map_err(|_| ParseError::Error)
}

fn parameter(buf: &[u8]) -> ParseResult<Parameter> {
    if buf.len() < 4 {
        return Err(ParseError::NeedData(4 - buf.len()));
    }
    let mut p = 0;
    for c in &buf[0..4] {
        if !c.is_ascii_digit() {
            return Err(ParseError::Error);
        }
        p *= 10;
        p += c - b'0';
    }
    Parameter::try_from(p as usize)
        .map(|p| (&buf[4..], p))
        .map_err(|_| ParseError::Error)
}

fn value_etx(buf: &[u8]) -> ParseResult<Value> {
    let end = buf
        .iter()
        .position(|c| c == &ETX)
        .ok_or(ParseError::NeedData(1))?;
    if end > 6 {
        // Max length of value
        return Err(ParseError::Error);
    }
    let val = Value::try_from(&buf[0..end]).map_err(|_| ParseError::Error)?;
    Ok((&buf[end..], val))
}
