mod utils;

use utils::{format_time, format_path};
use log::{Level, LevelFilter, Log, Metadata, Record, SetLoggerError};
use ansi_term::Color;

pub use utils::{Timezone};

static LOG_LEVEL_NAMES: [&str; 6] = ["N/A", "E", "W", "I", "D", "T"];

fn level_to_str(l: &Level) -> &'static str {
    LOG_LEVEL_NAMES[*l as usize]
}


#[derive(Clone)]
pub struct Options {
    pub level: LevelFilter,
    pub print_level: bool,
    pub timezone: Timezone,
    pub colored: bool,
}

impl Default for Options {
    fn default() -> Self {
        Self {
            level: LevelFilter::Trace,
            print_level: true,
            timezone: Timezone::Local,
            colored: true,
        }
    }
}

pub struct XLog {
    level: LevelFilter,
    print_level: bool,
    timezone: Timezone,
    colored: bool,
}

impl XLog {
    #[must_use = "You must call init() to begin logging"]
    pub fn new(opts: Options) -> XLog {
        let Options {
            level, print_level, timezone, colored
        } = opts;
        XLog {
            level,
            print_level,
            timezone,
            colored,
        }
    }

    // pub fn with_level(mut self, level: LevelFilter) -> XLog {
    //     self.level = level;
    //     self
    // }
    //
    // pub fn with_timezone(mut self, timezone: Timezone) -> XLog {
    //     self.timezone = timezone;
    //     self
    // }
    //
    // pub fn with_color(mut self, colored: bool) -> XLog {
    //     self.colored = colored;
    //     self
    // }
    //
    // pub fn print_level(mut self, print_level: bool) -> XLog {
    //     self.print_level = print_level;
    //     self
    // }

    pub fn init(self) -> Result<(), SetLoggerError> {
        log::set_max_level(log::STATIC_MAX_LEVEL);
        log::set_boxed_logger(Box::new(self))?;
        Ok(())
    }
}

impl Default for XLog {
    fn default() -> Self {
        XLog::new(Options::default())
    }
}

impl Log for XLog {
    fn enabled(&self, metadata: &Metadata) -> bool {
        metadata.level().to_level_filter() <= self.level
    }

    fn log(&self, record: &Record) {
        if self.enabled(record.metadata()) {
            let timestamp = format_time(&self.timezone);

            let mut pathname = format_path(record.file().unwrap_or("???"),
                                           record.line().unwrap_or(0));

            let level_string = match self.print_level {
                true => format!("[{}] ", level_to_str(&record.level())),
                false => "".to_string()
            };

            let mut color_msg = format!("{}{}", level_string, record.args());

            if self.colored {
                // timestamp = Color::Fixed(250).paint(timestamp).to_string();
                pathname = Color::Blue.paint(pathname).to_string();

                color_msg = match record.level() {
                    Level::Error => Color::Red.paint(color_msg).to_string(),
                    Level::Warn => Color::Yellow.paint(color_msg).to_string(),
                    Level::Info => Color::Green.paint(color_msg).to_string(),
                    Level::Debug => Color::Blue.paint(color_msg).to_string(),
                    Level::Trace => Color::White.paint(color_msg).to_string(),
                };
            }

            let message = format!("{}{} {}", timestamp, pathname, color_msg);

            #[cfg(not(feature = "stderr"))]
            println!("{}", message);

            #[cfg(feature = "stderr")]
            eprintln!("{}", message);
        }
    }

    fn flush(&self) {}
}

use std::sync::Once;

static INIT: Once = Once::new();

pub fn init_once(opts: Option<Options>) {
    INIT.call_once(|| {
        let opts = opts.unwrap_or_default();
        XLog::new(opts).init().unwrap();
    });
}

// pub use log::{trace, debug, info, warn, error};

#[macro_export]
macro_rules! trace {
    // (target: $target:expr, $($arg:tt)+) => ({
    //     log::log!(target: $target, log::Level::Trace, $($arg)+)
    // });
    ($($arg:tt)*) => ({
        x_log::init_once(None);
        log::log!(log::Level::Trace, $($arg)+);
    });
}

#[macro_export]
macro_rules! debug {
    ($($arg:tt)*) => ({
        x_log::init_once(None);
        log::log!(log::Level::Debug, $($arg)+);
    });
}

#[macro_export]
macro_rules! info {
    ($($arg:tt)*) => ({
        x_log::init_once(None);
        log::log!(log::Level::Info, $($arg)+);
    });
}

#[macro_export]
macro_rules! warn {
    ($($arg:tt)*) => ({
        x_log::init_once(None);
        log::log!(log::Level::Warn, $($arg)+);
    });
}

#[macro_export]
macro_rules! error {
    ($($arg:tt)*) => ({
        x_log::init_once(None);
        log::log!(log::Level::Error, $($arg)+);
    });
}
////////////////////////////////////////////////////////////////////////////////
pub use utils::__print_val;

#[macro_export]
macro_rules! __val {
    ($fmt:expr, $($arg:tt)+) => (
        crate::__print_val(file!(), line!(), format_args!($fmt, $($arg)+));
    )
}

#[macro_export]
macro_rules! val {
    ($v:expr) => {
        __val!("{}︎︎︎︎⏤►{:?}", stringify!($v), $v);
    };
    ($v:expr, $v2:expr) => {
        __val!("{}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}", stringify!($v), $v, stringify!($v2), $v2,);
    };
    ($v:expr, $v2:expr, $v3:expr) => {
        __val!("{}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3);
    };
    ($v:expr, $v2:expr, $v3:expr, $v4:expr) => {
        __val!("{}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3, stringify!($v4), $v4);
    };
    ($v:expr, $v2:expr, $v3:expr, $v4:expr, $v5:expr) => {
        __val!("{}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3, stringify!($v4), $v4, stringify!($v5), $v5);
    };
    ($v:expr, $v2:expr, $v3:expr, $v4:expr, $v5:expr, $v6:expr) => {
        __val!("{}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}, {}︎︎︎︎⏤►{:?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3, stringify!($v4), $v4, stringify!($v5), $v5, stringify!($v6), $v6);
    };
}

#[macro_export]
macro_rules! valn {
    ($v:expr) => {
        __val!("\n{}︎︎︎︎⏤►{:?}", stringify!($v), $v);
    };
    ($v:expr, $v2:expr) => {
        __val!("\n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?}", stringify!($v), $v, stringify!($v2), $v2,);
    };
    ($v:expr, $v2:expr, $v3:expr) => {
        __val!("\n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3);
    };
    ($v:expr, $v2:expr, $v3:expr, $v4:expr) => {
        __val!("\n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3, stringify!($v4), $v4);
    };
    ($v:expr, $v2:expr, $v3:expr, $v4:expr, $v5:expr) => {
        __val!("\n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3, stringify!($v4), $v4, stringify!($v5), $v5);
    };
    ($v:expr, $v2:expr, $v3:expr, $v4:expr, $v5:expr, $v6:expr) => {
        __val!("\n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?} \n{}︎︎︎︎⏤►{:?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3, stringify!($v4), $v4, stringify!($v5), $v5, stringify!($v6), $v6);
    };
}

#[macro_export]
macro_rules! valv {
    ($v:expr) => {
        __val!("\n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?}", stringify!($v), $v);
    };
    ($v:expr, $v2:expr) => {
        __val!("\n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?}", stringify!($v), $v, stringify!($v2), $v2,);
    };
    ($v:expr, $v2:expr, $v3:expr) => {
        __val!("\n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3);
    };
    ($v:expr, $v2:expr, $v3:expr, $v4:expr) => {
        __val!("\n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3, stringify!($v4), $v4);
    };
    ($v:expr, $v2:expr, $v3:expr, $v4:expr, $v5:expr) => {
        __val!("\n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3, stringify!($v4), $v4, stringify!($v5), $v5);
    };
    ($v:expr, $v2:expr, $v3:expr, $v4:expr, $v5:expr, $v6:expr) => {
        __val!("\n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?} \n⏤⏤ {}︎︎︎︎ ⏤⏤\n{:#?}", stringify!($v), $v, stringify!($v2), $v2, stringify!($v3), $v3, stringify!($v4), $v4, stringify!($v5), $v5, stringify!($v6), $v6);
    };
}