use ring::{rand, signature};
use std::{io::{BufReader, Read}, path::Path};

use crate::{SdkResult, error::SdkError, utils::base64};

pub fn sign_sha256_with_rsa(source: String, key_path: &Path) -> SdkResult<String> {
    let keyfile =
        std::fs::File::open(key_path).map_err(|e| SdkError::MsgDecryptError(e.to_string()))?;
    let mut reader = BufReader::new(keyfile);
    let keys = rustls_pemfile::pkcs8_private_keys(&mut reader).unwrap();
    assert!(keys.len() == 1);
    let private_key = keys[0].clone();
    // let mut private_key: Vec<u8> = Vec::new();
    // keyfile.read_to_end(&mut private_key)
    //     .map_err(|e| SdkError::MsgDecryptError(e.to_string()))?;

    let key_pair = signature::RsaKeyPair::from_pkcs8(&private_key)
        .map_err(|e| SdkError::MsgDecryptError(e.to_string()))?;
    let rng = rand::SystemRandom::new();
    let mut signature = vec![0; key_pair.public_modulus_len()];
    key_pair
        .sign(
            &signature::RSA_PKCS1_SHA256,
            &rng,
            source.as_bytes(),
            &mut signature,
        )
        .map_err(|e| SdkError::MsgDecryptError(e.to_string()))?;
    Ok(base64::encode(&signature))
}

#[cfg(test)]
mod tests {
    use crate::pay::utils::sign::sign_sha256_with_rsa;
    use std::{error::Error, path::Path};

    #[test]
    fn test_decrypt() -> Result<(), Box<dyn Error>> {
        let source = "source";
        // openssl pkcs8 -topk8 -nocrypt -outform der -in test_private_key.pem > rsa-2048-private-key.pk8
        // let key_path = Path::new("./src/pay/utils/rsa-2048-private-key.pk8");
        let key_path = Path::new("./src/pay/utils/test_private_key.pem");
        let want = "BKyAfU4iMCuvXMXS0Wzam3V/cnxZ+JaqigPM5OhljS2iOT95OO6Fsuml2JkFANJU9K6q9bLlDhPXuoVz+pp4hAm6pHU4ld815U4jsKu1RkyaII+1CYBUYC8TK0XtJ8FwUXXz8vZHh58rrAVN1XwNyvD1vfpxrMT4SL536GLwvpUHlCqIMzoZUguLli/K8V29QiOhuH6IEqLNJn8e9b3nwNcQ7be3CzYGpDAKBfDGPCqCv8Rw5zndhlffk2FEA70G4hvMwe51qMN/RAJbknXG23bSlObuTCN7Ndj1aJGH6/L+hdwfLpUtJm4QYVazzW7DFD27EpSQEqA8bX9+8m1rLg==";
        let ret = sign_sha256_with_rsa(source.to_string(), &key_path)?;
        assert_eq!(ret, want);
        Ok(())
    }
}
