use ring::aead::{
    Aad, Algorithm, BoundKey, LessSafeKey, Nonce, OpeningKey, UnboundKey, AES_128_GCM, AES_256_GCM,
};

use crate::{error::SdkError, utils::base64, SdkResult};

pub fn decrypt_aes256_gcm(
    aes_key: String,
    associated_data: String,
    nonce: String,
    cipher_text: String,
) -> SdkResult<String> {
    let mut cipher_text =
        base64::decode(cipher_text).map_err(|e| SdkError::MsgDecryptError(e.to_string()))?;
    let aad = Aad::from(associated_data);
    let nonce = Nonce::try_assume_unique_for_key(nonce.as_bytes())
        .map_err(|e| SdkError::MsgDecryptError(e.to_string()))?;
    let key = UnboundKey::new(&AES_256_GCM, aes_key.as_bytes())
        .map_err(|e| SdkError::MsgDecryptError(e.to_string()))?;
    let key = LessSafeKey::new(key);
    let ret: _ = key
        .open_in_place(nonce, aad, &mut cipher_text)
        .map_err(|e| SdkError::MsgDecryptError(e.to_string()))?;
    Ok(String::from_utf8_lossy(ret).to_string())
}

#[cfg(test)]
mod tests {
    use rand::rngs::adapter::ReseedingRng;
    use std::error::Error;

    use crate::pay::utils::aes::decrypt_aes256_gcm;

    #[test]
    fn test_decrypt() -> Result<(), Box<dyn Error>> {
        // let testAESUtilAPIV3Key       = "testAPIv3Key0000".to_string();
        // let testAESUtilCiphertext     = "FsdXzxryWfKwvLJKf8LG/ToRPTRh8RN9wROC".to_string();
        // let testAESUtilPlaintext      = "Hello World".to_string();
        // let testAESUtilNonce          = "wCq51qZv4Yfg".to_string();
        // let testAESUtilAssociatedData = "Dl56FrqWQJF1t9LtC3vEUsniZKvbqdR8".to_string();
        // let ret= decrypt_aes256_gcm(testAESUtilAPIV3Key, testAESUtilAssociatedData, testAESUtilNonce, testAESUtilCiphertext);
        // assert_eq!(testAESUtilPlaintext, ret);
        Ok(())
    }
}
