use std::process::exit;

use clap::Parser;
use wpa_psk::{bytes_to_hex, wpa_psk, wpa_psk_unchecked, Passphrase, Ssid};

/// Compute the WPA-PSK of a Wi-Fi SSID and passphrase.
#[derive(Debug, Parser)]
#[clap(about, version, author)]
struct Args {
    /// An SSID consisting of 1 up to 32 arbitrary bytes.
    ssid: String,

    /// A passphrase consisting of 8 up to 63 printable ASCII characters.
    passphrase: String,

    /// Don't check the given SSID and passphrase.
    #[clap(short, long)]
    force: bool,
}

fn main() {
    exit(match run() {
        Ok(_) => 0,
        Err(err) => {
            eprintln!("{}", err);
            1
        }
    })
}

fn run() -> Result<(), Box<dyn std::error::Error>> {
    let args = Args::try_parse()?;
    let psk = if args.force {
        wpa_psk_unchecked(args.ssid.as_bytes(), args.passphrase.as_bytes())
    } else {
        let ssid = Ssid::try_from(args.ssid.as_bytes())?;
        let passphrase = Passphrase::try_from(args.passphrase.as_bytes())?;
        wpa_psk(&ssid, &passphrase)
    };
    println!("0x{}", bytes_to_hex(&psk));
    Ok(())
}

#[test]
fn verify_clap_app() {
    use clap::IntoApp;
    Args::into_app().debug_assert()
}
