use serde_derive::{Deserialize, Serialize};

#[derive(Default, Debug, Clone, PartialEq, Serialize, Deserialize)]
#[serde(rename_all = "camelCase")]
pub struct Organization {
    pub id: String,
    pub object: String,
    pub name: String,
    pub domains: Vec<Domain>,
}

#[derive(Default, Debug, Clone, PartialEq, Serialize, Deserialize)]
#[serde(rename_all = "camelCase")]
pub struct Domain {
    pub domain: String,
    pub id: String,
    pub object: String,
}

#[cfg(test)]
mod tests {
    use crate::client::Client;
    use expect_test::expect;
    use httpmock::Method::GET;
    use httpmock::MockServer;

    const TEST_API_KEY: &str = "TEST_API_KEY";

    #[async_std::test]
    async fn get_organization() -> Result<(), Box<dyn std::error::Error>> {
        const ORGANIZATION_ID: &str = "org_01EHZNVPK3SFK441A1RGBFSHRT";
        let server = MockServer::start();

        let client = Client::new(Some(server.base_url()), TEST_API_KEY.to_string());

        let mock = server.mock(|when, then| {
            when.method(GET)
                .path(format!("/organizations/{}", ORGANIZATION_ID))
                .header("Authorization", &format!("Bearer {}", TEST_API_KEY));
            then.status(200)
                .header("Content-Type", mime::JSON.as_str())
                .body(r#"{ "id": "org_01EHZNVPK3SFK441A1RGBFSHRT", "object": "organization", "name": "Foo Corp", "domains": [ { "id": "org_domain_01EHZNVPK2QXHMVWCEDQEKY69A", "object": "organization_domain", "domain": "foo-corp.com" } ] }"#);
        });

        let organization = client.get_organization(ORGANIZATION_ID).await?;

        mock.assert();

        expect![[r#"
            Organization {
                id: "org_01EHZNVPK3SFK441A1RGBFSHRT",
                object: "organization",
                name: "Foo Corp",
                domains: [
                    Domain {
                        domain: "foo-corp.com",
                        id: "org_domain_01EHZNVPK2QXHMVWCEDQEKY69A",
                        object: "organization_domain",
                    },
                ],
            }
        "#]]
        .assert_debug_eq(&organization);

        Ok(())
    }
}
