use serde_derive::{Deserialize, Serialize};
use serde_json::{Map, Value};

#[derive(Default, Debug, Clone, PartialEq, Serialize, Deserialize)]
#[serde(rename_all = "camelCase")]
pub struct User {
    pub id: String,
    #[serde(rename = "idp_id")]
    pub idp_id: String,
    pub emails: Vec<Email>,
    #[serde(rename = "first_name")]
    pub first_name: String,
    #[serde(rename = "last_name")]
    pub last_name: String,
    pub username: String,
    pub groups: Vec<Group>,
    pub state: String,
    #[serde(rename = "raw_attributes")]
    pub raw_attributes: Option<Map<String, Value>>,
}

#[derive(Default, Debug, Clone, PartialEq, Serialize, Deserialize)]
#[serde(rename_all = "camelCase")]
pub struct Email {
    pub primary: bool,
    #[serde(rename = "type")]
    pub type_field: String,
    pub value: String,
}

#[derive(Default, Debug, Clone, PartialEq, Serialize, Deserialize)]
#[serde(rename_all = "camelCase")]
pub struct Group {
    pub id: String,
    pub name: String,
    #[serde(rename = "raw_attributes")]
    pub raw_attributes: Option<Map<String, Value>>,
}

#[cfg(test)]
mod tests {
    use crate::client::Client;
    use expect_test::expect;
    use httpmock::Method::GET;
    use httpmock::MockServer;

    const TEST_API_KEY: &str = "TEST_API_KEY";

    #[async_std::test]
    async fn get_user() -> Result<(), Box<dyn std::error::Error>> {
        const USER_ID: &str = "directory_user_01E1JG7J09H96KYP8HM9B0G5SJ";
        let server = MockServer::start();

        let client = Client::new(Some(server.base_url()), TEST_API_KEY.to_string());

        let mock = server.mock(|when, then| {
            when.method(GET)
                .path(format!("/directory_users/{}", USER_ID))
                .header("Authorization", &format!("Bearer {}", TEST_API_KEY));
            then.status(200)
                .header("Content-Type", mime::JSON.as_str())
                .body(r#"{"id":"directory_user_01E1JG7J09H96KYP8HM9B0G5SJ","idp_id":"2836","first_name":"Marcelina","last_name":"Davis","emails":[{"primary":true,"type":"work","value":"marcelina@foo-corp.com"}],"username":"marcelina@foo-corp.com","groups":[{"id":"directory_group_01E64QTDNS0EGJ0FMCVY9BWGZT","name":"Engineering"}],"state":"active"}"#);
        });

        let user = client.get_user(USER_ID).await?;

        mock.assert();

        expect![[r#"
            User {
                id: "directory_user_01E1JG7J09H96KYP8HM9B0G5SJ",
                idp_id: "2836",
                emails: [
                    Email {
                        primary: true,
                        type_field: "work",
                        value: "marcelina@foo-corp.com",
                    },
                ],
                first_name: "Marcelina",
                last_name: "Davis",
                username: "marcelina@foo-corp.com",
                groups: [
                    Group {
                        id: "directory_group_01E64QTDNS0EGJ0FMCVY9BWGZT",
                        name: "Engineering",
                        raw_attributes: None,
                    },
                ],
                state: "active",
                raw_attributes: None,
            }
        "#]]
        .assert_debug_eq(&user);

        Ok(())
    }

    #[async_std::test]
    async fn get_user_not_found() -> Result<(), Box<dyn std::error::Error>> {
        const USER_ID: &str = "workos_user_xxxx";
        let server = MockServer::start();

        let client = Client::new(Some(server.base_url()), TEST_API_KEY.to_string());

        let mock = server.mock(|when, then| {
            when.method(GET)
                .path(format!("/directory_users/{}", USER_ID))
                .header("Authorization", &format!("Bearer {}", TEST_API_KEY));
            then.status(404);
        });

        let res = client.get_user(USER_ID).await;

        mock.assert();

        expect![[r#"identifier `workos_user_xxxx` not found"#]]
            .assert_eq(&res.unwrap_err().to_string());

        Ok(())
    }

    #[async_std::test]
    async fn get_user_unknown_error() -> Result<(), Box<dyn std::error::Error>> {
        const USER_ID: &str = "workos_user_xxxx";
        let server = MockServer::start();

        let client = Client::new(Some(server.base_url()), TEST_API_KEY.to_string());

        let mock = server.mock(|when, then| {
            when.method(GET)
                .path(format!("/directory_users/{}", USER_ID))
                .header("Authorization", &format!("Bearer {}", TEST_API_KEY));
            then.status(500);
        });

        let res = client.get_user(USER_ID).await;

        mock.assert();

        expect![[r#"unknown error"#]].assert_eq(&res.unwrap_err().to_string());

        Ok(())
    }

    #[async_std::test]
    async fn get_group() -> Result<(), Box<dyn std::error::Error>> {
        const GROUP_ID: &str = "directory_group_01E1JJS84MFPPQ3G655FHTKX6Z";
        let server = MockServer::start();

        let client = Client::new(Some(server.base_url()), TEST_API_KEY.to_string());

        let mock = server.mock(|when, then| {
            when.method(GET)
                .path(format!("/directory_groups/{}", GROUP_ID))
                .header("Authorization", &format!("Bearer {}", TEST_API_KEY));
            then.status(200)
                .header("Content-Type", mime::JSON.as_str())
                .body(r#"{"id":"directory_group_01E1JJS84MFPPQ3G655FHTKX6Z","name":"Developers"}"#);
        });

        let group = client.get_group(GROUP_ID).await?;

        mock.assert();

        expect![[r#"
            Group {
                id: "directory_group_01E1JJS84MFPPQ3G655FHTKX6Z",
                name: "Developers",
                raw_attributes: None,
            }
        "#]]
        .assert_debug_eq(&group);

        Ok(())
    }
}
