use woodland::binary_tree::BinaryTree;
use woodland::prelude::*;

fn build_two_level_tree() -> BinaryTree<u8> {
    /* Tree looks like the following:
     *
     *              4
     *          /       \
     *         2         6
     *       /   \     /   \
     *      1     3   5     7
     */

    let mut new_tree = BinaryTree::<u8>::new();
    let left_left_id = new_tree.create_node(1);
    let left_right_id = new_tree.create_node(3);
    let left_id = new_tree.create_node(2);
    let right_id = new_tree.create_node(6);
    let right_left_id = new_tree.create_node(5);
    let right_right_id = new_tree.create_node(7);
    new_tree.root = Some(new_tree.create_node(4));

    let root_ref = new_tree.get_node_mut(new_tree.root.unwrap()).unwrap();
    root_ref.set_left(Some(left_id));
    root_ref.set_right(Some(right_id));

    let left_ref = new_tree.get_node_mut(left_id).unwrap();
    left_ref.set_left(Some(left_left_id));
    left_ref.set_right(Some(left_right_id));

    let right_ref = new_tree.get_node_mut(right_id).unwrap();
    right_ref.set_left(Some(right_left_id));
    right_ref.set_right(Some(right_right_id));

    new_tree
}

#[test]
fn tree_new_is_empty() {
    let new_tree = BinaryTree::<u8>::new();
    assert!(new_tree.is_empty());
}

#[test]
fn mutating_root_works() {
    let mut new_tree = BinaryTree::<u8>::new();
    let root_id = new_tree.create_node(42);
    new_tree.root = Some(root_id);
    let root_ref = new_tree.get_node_mut(new_tree.root.unwrap()).unwrap();
    let root_data = root_ref.get_data_mut();
    *root_data = 10 as u8;

    assert_eq!(
        *new_tree
            .get_node(new_tree.root.unwrap())
            .unwrap()
            .get_data(),
        10 as u8
    );
}

#[test]
fn simple_tree() {
    let mut new_tree = BinaryTree::<u8>::new();
    let left_id = new_tree.create_node(10);
    let right_id = new_tree.create_node(5);
    new_tree.root = Some(new_tree.create_node(42));
    let root_ref = new_tree.get_node_mut(new_tree.root.unwrap()).unwrap();
    root_ref.set_left(Some(left_id));
    root_ref.set_right(Some(right_id));

    assert_eq!(root_ref.get_left(), Some(left_id));
    assert_eq!(root_ref.get_right(), Some(right_id));
    assert_eq!(root_ref.get_data(), &42);
}

#[test]
fn preorder_iterator_works() {
    let tree = build_two_level_tree();
    let preorder: Vec<&u8> = tree.preorder_iter().map(|v| v.get_data()).collect();
    assert_eq!(preorder, vec![&4, &2, &1, &3, &6, &5, &7]);
}

#[test]
fn inorder_iterator_works() {
    let tree = build_two_level_tree();
    let inorder: Vec<&u8> = tree.inorder_iter().map(|v| v.get_data()).collect();
    assert_eq!(inorder, vec![&1, &2, &3, &4, &5, &6, &7]);
}

#[test]
fn postorder_iterator_works() {
    let tree = build_two_level_tree();
    let inorder: Vec<&u8> = tree.postorder_iter().map(|v| v.get_data()).collect();
    assert_eq!(inorder, vec![&1, &3, &2, &5, &7, &6, &4]);
}
