use crate::node::{NodeId, NodeOperations};

/// Defines all the common operations that must be supported for any tree.
///
/// The type of the object the tree is going to be containing (Nodes) must implement
/// the [`NodeOperations`] trait.
pub trait TreeOperations<NodeType>
where
    NodeType: NodeOperations,
{
    /// Create a new instance of the tree with an empty root.
    fn new() -> Self;

    /// Genereate a new mutable reference to a NodeType. You must call this function to get new references
    /// since we are allocating all references in the Arena.
    fn create_node(&mut self, data: NodeType::T) -> NodeId;

    /// Get the number of nodes in the tree.
    fn count(&self) -> usize;

    /// Returns `true` if there are no nodes in the tree, and returns `false` otherwise.
    fn is_empty(&self) -> bool;

    /// Get a refrence to a `NodeType<T>` from a given `NodeId`, if a node with the specified
    /// `NodeId` exists in the tree.
    fn get_node(&self, node_id: NodeId) -> Option<&NodeType>;

    /// Get a mutable reference to a `NodeType<T>` from a given `NodeId`, if a node with the
    /// specified `NodeId` exists in the tree.
    fn get_node_mut(&mut self, node_id: NodeId) -> Option<&mut NodeType>;

    /// Search to see if a node is inside the tree.
    fn search(&self, node: NodeId) -> bool;

    /// Insert a node into the tree.
    fn insert(&mut self, _node: NodeId);
}

/// Defines all the iterator traversal operations that must be supported to be defined as an
/// iterative tree.
pub trait IterativeTreeOperations<NodeType>
where
    NodeType: NodeOperations,
{
    /// Get a preorder traveral iterator.
    fn preorder_iter(&self) -> Box<dyn Iterator<Item = &NodeType> + '_>;

    /// Get an inorder traversal iterator.
    fn inorder_iter(&self) -> Box<dyn Iterator<Item = &NodeType> + '_>;

    /// Get a postorder traversal iterator.
    fn postorder_iter(&self) -> Box<dyn Iterator<Item = &NodeType> + '_>;
}
