//! Implementation of a simple Binary Tree, which has no invariants to maintain.

use std::collections::HashMap;

use crate::node::{BinaryNodeOperations, NodeId, NodeOperations};
use crate::tree::{IterativeTreeOperations, TreeOperations};

/// The most basic tree in `woodland`. All nodes stored in the tree are [`BinaryTreeNode<T>`]s.
///
/// **NOTE:** The [`TreeOperations::insert()`] definition for this tree will call [`std::unimplemented`]
/// as there is no invariant to fall back on to determine where to insert the node.
///
/// Example usage:
/// ```
/// use woodland::prelude::*;
/// use woodland::binary_tree::BinaryTree;
///
/// /* Tree looks like the following:
///  *
///  *              4
///  *          /       \
///  *         2         6
///  *       /   \     /   \
///  *      1     3   5     7
///  */
///
/// let mut new_tree = BinaryTree::<u8>::new();
/// let left_left_id = new_tree.create_node(1);
/// let left_right_id = new_tree.create_node(3);
/// let left_id = new_tree.create_node(2);
/// let right_id = new_tree.create_node(6);
/// let right_left_id = new_tree.create_node(5);
/// let right_right_id = new_tree.create_node(7);
/// new_tree.root = Some(new_tree.create_node(4));
///
/// let root_ref = new_tree.get_node_mut(new_tree.root.unwrap()).unwrap();
/// root_ref.set_left(Some(left_id));
/// root_ref.set_right(Some(right_id));
///
/// let left_ref = new_tree.get_node_mut(left_id).unwrap();
/// left_ref.set_left(Some(left_left_id));
/// left_ref.set_right(Some(left_right_id));
///
/// let right_ref = new_tree.get_node_mut(right_id).unwrap();
/// right_ref.set_left(Some(right_left_id));
/// right_ref.set_right(Some(right_right_id));
/// ```
pub struct BinaryTree<T>
where
    T: std::hash::Hash,
{
    /// The root node's [`NodeId`].
    pub root: Option<NodeId>,
    arena: HashMap<NodeId, BinaryTreeNode<T>>,
}

impl<T> BinaryTree<T>
where
    T: std::hash::Hash,
{
    fn get_next_node_id(&self) -> NodeId {
        NodeId::new(self.arena.len())
    }
}

impl<T> TreeOperations<BinaryTreeNode<T>> for BinaryTree<T>
where
    T: std::hash::Hash,
    T: Eq,
    T: PartialEq,
{
    fn new() -> Self {
        Self {
            root: None,
            arena: HashMap::new(),
        }
    }

    fn create_node(&mut self, data: T) -> NodeId {
        let new_id = self.get_next_node_id();
        let new_node = BinaryTreeNode::<T> {
            data,
            id: new_id,
            left: None,
            right: None,
        };

        self.arena.insert(new_id, new_node);
        new_id
    }

    fn count(&self) -> usize {
        self.arena.len()
    }

    fn is_empty(&self) -> bool {
        self.arena.is_empty()
    }

    fn get_node(&self, node_id: NodeId) -> Option<&BinaryTreeNode<T>> {
        self.arena.get(&node_id)
    }

    fn get_node_mut(&mut self, node_id: NodeId) -> Option<&mut BinaryTreeNode<T>> {
        self.arena.get_mut(&node_id)
    }

    fn search(&self, node_id: NodeId) -> bool {
        self.arena.contains_key(&node_id)
    }

    fn insert(&mut self, _node: NodeId) {
        unimplemented!("No invariant to uphold for BinaryTrees, so we don't know where to insert!");
    }
}

impl<'a, T> IterativeTreeOperations<BinaryTreeNode<T>> for BinaryTree<T>
where
    T: std::hash::Hash,
    T: Eq,
    T: PartialEq,
{
    fn preorder_iter(&self) -> Box<dyn Iterator<Item = &BinaryTreeNode<T>> + '_> {
        let mut preorder_vec: Vec<&BinaryTreeNode<T>> = Vec::new();

        if self.root.is_some() {
            let mut traversal_stack: Vec<&BinaryTreeNode<T>> =
                vec![self.get_node(self.root.unwrap()).unwrap()];

            while !traversal_stack.is_empty() {
                let curr = traversal_stack.pop().unwrap();
                preorder_vec.push(curr);

                /* Push right and left children */
                if let Some(right_id) = curr.get_right() {
                    traversal_stack.push(self.get_node(right_id).unwrap());
                }

                if let Some(left_id) = curr.get_left() {
                    traversal_stack.push(self.get_node(left_id).unwrap());
                }
            }
        }

        Box::new(preorder_vec.into_iter())
    }

    fn inorder_iter(&self) -> Box<dyn Iterator<Item = &BinaryTreeNode<T>> + '_> {
        let mut inorder_vec: Vec<&BinaryTreeNode<T>> = Vec::new();

        if self.root.is_some() {
            let mut traversal_stack: Vec<&BinaryTreeNode<T>> = Vec::new();
            let mut curr = self.get_node(self.root.unwrap());
            while curr.is_some() || !traversal_stack.is_empty() {
                /* Reach the left most Node of the curr node */
                while curr.is_some() {
                    traversal_stack.push(curr.unwrap());
                    if let Some(left_id) = curr.unwrap().get_left() {
                        curr = self.get_node(left_id);
                    } else {
                        curr = None;
                    }
                }

                /* Current must be None at this point */
                curr = traversal_stack.pop();

                inorder_vec.push(curr.unwrap());

                /* We have visited the node and its left subtree. Now it's right subtree's turn */
                if let Some(right_id) = curr.unwrap().get_right() {
                    curr = self.get_node(right_id);
                } else {
                    curr = None;
                }
            }
        }

        Box::new(inorder_vec.into_iter())
    }

    fn postorder_iter(&self) -> Box<dyn Iterator<Item = &BinaryTreeNode<T>> + '_> {
        let mut postorder_vec: Vec<&BinaryTreeNode<T>> = Vec::new();

        if self.root.is_some() {
            let mut traversal_stack: Vec<&BinaryTreeNode<T>> = Vec::new();
            let mut curr = self.get_node(self.root.unwrap());
            let mut pre: Option<NodeId> = None;

            while curr.is_some() || !traversal_stack.is_empty() {
                if curr.is_some() {
                    traversal_stack.push(curr.unwrap());
                    if let Some(left_id) = curr.unwrap().get_left() {
                        curr = self.get_node(left_id);
                    } else {
                        curr = None;
                    }
                } else {
                    curr = traversal_stack.pop();
                    traversal_stack.push(curr.unwrap());

                    if curr.unwrap().get_right().is_none()
                        || curr.unwrap().get_right().unwrap() == pre.unwrap()
                    {
                        postorder_vec.push(curr.unwrap());
                        traversal_stack.pop();
                        pre = Some(curr.unwrap().get_id());
                        curr = None;
                    } else if let Some(right_id) = curr.unwrap().get_right() {
                        curr = self.get_node(right_id);
                    } else {
                        curr = None;
                    }
                }
            }
        }

        Box::new(postorder_vec.into_iter())
    }
}

/// The most simple node type in `woodland`. Stores a data field, a [`NodeId`], as well as a `left` and `right`
/// child.
///
///
/// All get's and set's are implemented via the [`NodeOperations`] and [`BinaryNodeOperations`]
/// traits.
///
///
/// Notice that there is no way to create a [`BinaryTreeNode`] explicitly, as all nodes **must** be created by
/// the corresponding tree's memory arena. To learn more about memory arena's and the
/// implementation of tree's in `woodland`, refer to the documentation home page.
#[derive(Eq, PartialEq, Hash, Debug)]
pub struct BinaryTreeNode<T> {
    data: T,
    id: NodeId,
    left: Option<NodeId>,
    right: Option<NodeId>,
}

impl<T> NodeOperations for BinaryTreeNode<T> {
    type T = T;

    fn new(data: T, id: NodeId) -> Self {
        Self {
            data,
            id,
            left: None,
            right: None,
        }
    }

    fn get_id(&self) -> NodeId {
        self.id
    }

    fn get_data(&self) -> &T {
        &self.data
    }

    fn get_data_mut(&mut self) -> &mut T {
        &mut self.data
    }
}

impl<T> BinaryNodeOperations for BinaryTreeNode<T> {
    fn get_left(&self) -> Option<NodeId> {
        self.left
    }

    fn get_right(&self) -> Option<NodeId> {
        self.right
    }

    fn set_left(&mut self, left_node_id: Option<NodeId>) {
        self.left = left_node_id;
    }

    fn set_right(&mut self, right_node_id: Option<NodeId>) {
        self.right = right_node_id;
    }
}
