use crate::core::models::{MediaId, Message, Project, Version};
use crate::db::entities::prelude::*;
use anyhow::{bail, Result};
use sea_orm::prelude::*;
use sea_orm::DatabaseConnection;

pub async fn get_all_project(conn: &DatabaseConnection) -> Result<Vec<Project>> {
    let find_all_projects = ProjectEntity::find().all(conn).await?;
    let mut all_projects_list: Vec<Project> = Vec::new();
    for proj_model in find_all_projects.iter() {
        let project_data = Project {
            name: proj_model.name.to_owned(),
            desc: proj_model.description.to_owned(),
            url: proj_model.url.to_owned(),
            emoji: proj_model.emoji.to_owned(),
            featured: proj_model.is_featured,
            template: proj_model.is_template,
            tags: get_all_tags(conn, proj_model).await?,
            branch: get_branch(conn, proj_model).await?,
        };
        all_projects_list.push(project_data);
    }
    Ok(all_projects_list)
}

async fn get_branch(conn: &DatabaseConnection, proj_model: &ProjectModel) -> Result<String> {
    let find_branch = BranchEntity::find()
        .filter(BranchColumn::ProjectId.eq(&proj_model.name[..]))
        .one(conn)
        .await?;
    let branch_name = match find_branch {
        Some(m) => m.name.to_owned(),
        None => String::from("main"),
    };
    Ok(branch_name)
}

async fn get_all_tags(conn: &DatabaseConnection, proj_model: &ProjectModel) -> Result<Vec<String>> {
    let find_all_tags = TagEntity::find()
        .filter(TagColumn::ProjectId.eq(&proj_model.name[..]))
        .all(conn)
        .await?;
    let all_tags_list: Vec<String> = find_all_tags
        .iter()
        .map(|tag_model| tag_model.name.to_owned())
        .collect();
    Ok(all_tags_list)
}

pub async fn get_changelog_hash_ids(
    conn: &DatabaseConnection,
    project: &Project,
) -> Result<(String, String)> {
    let find_changelog = ChangelogEntity::find()
        .filter(ChangelogColumn::ProjectId.eq(&project.name[..]))
        .one(conn)
        .await?;
    let hash_ids = if let Some(changelog_model) = find_changelog {
        let prev_hash = changelog_model.prev_hash;
        let hash = changelog_model.hash;
        (prev_hash, hash)
    } else {
        bail!("fetch error: changelog not found in database")
    };

    Ok(hash_ids)
}

async fn get_hash_data(conn: &DatabaseConnection, hash_id: String) -> Result<String> {
    let find_hash = HashEntity::find_by_id(hash_id).one(conn).await;
    let hash_data = if let Some(hash_model) = find_hash? {
        hash_model.data
    } else {
        bail!("fetch error: hash data not found for given id")
    };
    Ok(hash_data)
}

// pub async fn get_all_version_hashes(
//     conn: &DatabaseConnection,
//     project: &Project,
// ) -> Result<Vec<String>> {
//     let find_versions = VersionEntity::find()
//         .filter(VersionColumn::ProjectId.eq(&project.name[..]))
//         .all(conn)
//         .await;

//     let mut all_version_hashes: Vec<String> = Vec::new();
//     for v in find_versions?.iter() {
//         all_version_hashes.push(v.hash.to_owned())
//     }

//     Ok(all_version_hashes)
// }

pub async fn get_all_versions(
    conn: &DatabaseConnection,
    project: &Project,
) -> Result<Vec<Version>> {
    let find_versions = VersionEntity::find()
        .filter(VersionColumn::ProjectId.eq(&project.name[..]))
        .all(conn)
        .await?;

    let mut all_version_data: Vec<Version> = Vec::new();
    for v in find_versions.iter() {
        let version_data = Version {
            major: v.major as u8,
            minor: v.minor as u8,
            patch: v.patch as u8,
            project_name: project.name.to_owned(),
            hash: v.hash.to_owned(),
            data: get_hash_data(conn, v.hash.to_owned()).await?,
            timestamp: v.timestamp.timestamp(),
            name: v.name.to_owned(),
            codename: v.codename.to_owned().unwrap_or_default(),
            start_commit: v.start_commit.to_owned().unwrap_or_default(),
            end_commit: v.end_commit.to_owned().unwrap_or_default(),
        };
        all_version_data.push(version_data)
    }

    Ok(all_version_data)
}

pub async fn get_message(
    conn: &DatabaseConnection,
    media: MediaId,
    version: &Version,
) -> Result<Message> {
    let platform_name: &str = media.into();
    let find_sns = SocialMediaEntity::find()
        .filter(SocialMediaColumn::PlatformName.eq(platform_name))
        .one(conn)
        .await;
    let sns_id = if let Some(sns_model) = find_sns? {
        sns_model.id
    } else {
        bail!("message fetch error: sns model not found")
    };
    let find_message = MessageEntity::find()
        .filter(MessageColumn::SocialMediaId.eq(sns_id))
        .filter(MessageColumn::VersionHash.eq(version.hash.as_str()))
        .one(conn)
        .await;
    let message_data = match find_message? {
        Some(m) => Message {
            media_id: media,
            project_name: version.project_name.clone(),
            version_hash: version.hash.clone(),
            timestamp: m.timestamp.timestamp(),
            message_id: m.message_id,
            message_hash: m.message_hash,
        },
        None => bail!("fetch error: sent message for given version and platform not found"),
    };
    Ok(message_data)
}
