use crate::core::models::{Changelog, MediaId, Message, Project, Version};
use crate::db::entities::prelude::*;
use anyhow::{bail, Result};
use chrono::prelude::*;
use sea_orm::prelude::*;
use sea_orm::{DatabaseConnection, Set};
use strum::IntoEnumIterator;

pub async fn insert_or_ignore_project(conn: &DatabaseConnection, project: &Project) -> Result<()> {
    // if project exists do not overwrite the entry
    let find_project = ProjectEntity::find()
        .filter(ProjectColumn::Name.contains(&project.name[..]))
        .one(conn)
        .await;
    match find_project? {
        Some(project_model) => println!("found project: {:?}, ignoring...", project_model),
        None => insert_project(conn, project).await?,
    };
    Ok(())
}
async fn insert_project(conn: &DatabaseConnection, proj: &Project) -> Result<()> {
    let project_data = ProjectActiveModel {
        name: Set(proj.name.clone()),
        description: Set(proj.desc.clone()),
        url: Set(proj.url.clone()),
        emoji: Set(proj.emoji.clone()),
        is_featured: Set(proj.featured),
        is_template: Set(proj.template),
        ..Default::default()
    };
    ProjectEntity::insert(project_data).exec(conn).await?;

    // Add project tags
    insert_tags(conn, proj).await?;
    // Add project branch list
    insert_branch(conn, proj).await?;

    Ok(())
}

async fn insert_tags(conn: &DatabaseConnection, project: &Project) -> Result<()> {
    let mut tags_model_list: Vec<TagActiveModel> = Vec::new();
    for tag in project.tags.iter() {
        let tag_data = TagActiveModel {
            name: Set(tag.clone()),
            project_id: Set(project.name.clone()),
            ..Default::default()
        };
        tags_model_list.push(tag_data);
    }
    TagEntity::insert_many(tags_model_list).exec(conn).await?;
    Ok(())
}

async fn insert_branch(conn: &DatabaseConnection, project: &Project) -> Result<()> {
    let branch_data = BranchActiveModel {
        name: Set(project.branch.clone()),
        project_id: Set(project.name.clone()),
        ..Default::default()
    };
    BranchEntity::insert(branch_data).exec(conn).await?;
    Ok(())
}

pub async fn insert_or_ignore_changelog(
    conn: &DatabaseConnection,
    changelog: &Changelog,
) -> Result<()> {
    // if project exists do not overwrite the entry
    let find_changelog = ChangelogEntity::find()
        .filter(ChangelogColumn::ProjectId.eq(&changelog.project_name[..]))
        .one(conn)
        .await;
    match find_changelog? {
        Some(changelog_model) => println!("found changelog {:?}, ignoring...", changelog_model),
        None => insert_changelog(conn, changelog).await?,
    };
    Ok(())
}

async fn insert_changelog(conn: &DatabaseConnection, changelog: &Changelog) -> Result<()> {
    // Insert hash data
    let hash_data = HashActiveModel {
        id: Set(changelog.hash.clone()),
        data: Set(changelog.data.clone()),
        ..Default::default()
    };
    HashEntity::insert(hash_data).exec(conn).await?;

    // Insert changelog data
    let changelog_data = ChangelogActiveModel {
        project_id: Set(changelog.project_name.clone()),
        hash: Set(changelog.hash.clone()),
        prev_hash: Set(changelog.hash.clone()),
        ..Default::default()
    };
    ChangelogEntity::insert(changelog_data).exec(conn).await?;

    Ok(())
}

async fn insert_hash_data(conn: &DatabaseConnection, hash_id: &str, hash_data: &str) -> Result<()> {
    let hash_data = HashActiveModel {
        id: Set(hash_id.to_string()),
        data: Set(hash_data.to_string()),
        ..Default::default()
    };
    HashEntity::insert(hash_data).exec(conn).await?;
    Ok(())
}

async fn insert_or_ignore_hash_data(
    conn: &DatabaseConnection,
    hash_id: &str,
    hash_data: &str,
) -> Result<()> {
    let find_hash = HashEntity::find()
        .filter(HashColumn::Id.eq(hash_id))
        .one(conn)
        .await?;
    match find_hash {
        Some(hash_model) => println!("found hash {:?}, ignoring...", hash_model),
        None => insert_hash_data(conn, hash_id, hash_data).await?,
    };
    Ok(())
}

pub async fn insert_or_ignore_versions(
    conn: &DatabaseConnection,
    versions: &Vec<Version>,
) -> Result<()> {
    let find_versions = VersionEntity::find()
        .filter(VersionColumn::ProjectId.eq(&versions[0].project_name[..]))
        .all(conn)
        .await?;

    if find_versions.len() != 0 {
        println!("found existing versions {:?}, ignoring...", find_versions)
    } else {
        insert_versions(conn, versions).await?;
    }
    Ok(())
}

async fn insert_versions(conn: &DatabaseConnection, versions: &Vec<Version>) -> Result<()> {
    let mut insert_all_versions: Vec<VersionActiveModel> = Vec::new();
    for version in versions.iter() {
        insert_or_ignore_hash_data(conn, version.hash.as_str(), version.data.as_str()).await?;
        let version_data = VersionActiveModel {
            project_id: Set(version.project_name.clone()),
            hash: Set(version.hash.clone()),
            timestamp: Set(chrono::Utc.timestamp(version.timestamp, 0)),
            name: Set(version.name.clone()),
            major: Set(version.major as i32),
            minor: Set(version.minor as i32),
            patch: Set(version.patch as i32),
            // codename: Set(Some(version.codename.clone())),
            codename: Set(None),
            start_commit: Set(Some(version.start_commit.clone())),
            end_commit: Set(Some(version.end_commit.clone())),
        };
        insert_all_versions.push(version_data);
    }
    VersionEntity::insert_many(insert_all_versions)
        .exec(conn)
        .await?;
    Ok(())
}

// async fn insert_version(conn: &DatabaseConnection, version: &Version) -> Result<()> {
//     insert_or_ignore_hash_data(conn, version.hash.as_str(), version.data.as_str()).await?;

//     let version_data = VersionActiveModel {
//         project_id: Set(version.project_name.clone()),
//         hash: Set(version.hash.clone()),
//         timestamp: Set(chrono::Utc.timestamp(version.timestamp, 0)),
//         name: Set(version.name.clone()),
//         major: Set(version.major),
//         minor: Set(version.minor),
//         patch: Set(version.patch),
//         // codename: Set(Some(version.codename.clone())),
//         codename: Set(None),
//         start_commit: Set(Some(version.start_commit.clone())),
//         end_commit: Set(Some(version.end_commit.clone())),
//     };
//     VersionEntity::insert(version_data).exec(conn).await?;
//     Ok(())
// }

pub async fn insert_sns_media(conn: &DatabaseConnection) -> Result<()> {
    let mut media_types_list: Vec<SocialMediaActiveModel> = Vec::new();
    for (idx, media_type) in MediaId::iter().enumerate() {
        let media_type_str: &str = media_type.into();
        let sns_model = SocialMediaActiveModel {
            id: Set(idx as i32),
            platform_name: Set(media_type_str.to_string()),
        };
        media_types_list.push(sns_model)
    }
    SocialMediaEntity::insert_many(media_types_list)
        .exec(conn)
        .await?;
    Ok(())
}
pub async fn insert_message(conn: &DatabaseConnection, message: &Message) -> Result<()> {
    let platform_name: &str = message.media_id.into();
    // find project id
    let find_sns = SocialMediaEntity::find()
        .filter(SocialMediaColumn::PlatformName.contains(platform_name))
        .one(conn)
        .await?;
    let sns_id = if let Some(sns_model) = find_sns {
        sns_model.id
    } else {
        bail!("social media {:?} not found in database", find_sns)
    };

    let message_data = MessageActiveModel {
        social_media_id: Set(sns_id),
        project_id: Set(message.project_name.clone()),
        version_hash: Set(message.version_hash.clone()),
        // timestamp: Set(message.timestamp),
        timestamp: Set(chrono::Utc.timestamp(message.timestamp, 0)),
        message_id: Set(message.message_id.clone()),
        message_hash: Set(message.message_hash.clone()),
        ..Default::default()
    };
    MessageEntity::insert(message_data).exec(conn).await?;

    Ok(())
}
