use crate::cache::memory::{
    get_all_projects, get_changelog_state, get_versions_for_project, APICacheConnection,
};
use crate::core::models::{Project, Response, StatefulChangelog, Version};
use rocket::serde::json::Json;
// use rocket::Request;

#[get("/")]
fn index() -> Json<Response> {
    Json(Response {
        ok: true,
        code: 200,
        message: "Woof woof :P, welcome to Wolfie!",
    })
}

#[get("/projects")]
async fn projects(conn: APICacheConnection) -> Json<Vec<Project>> {
    let all_projects = conn.run(|client| get_all_projects(client).unwrap()).await;
    Json(all_projects)
}

#[get("/project/<name>/log")]
async fn project_changelog(conn: APICacheConnection, name: String) -> Json<StatefulChangelog> {
    let proj_name = if name.is_ascii() {
        name.to_ascii_lowercase()
    } else {
        println!("invalid characters");
        String::default()
    };
    let project = Project {
        name: proj_name,
        ..Default::default()
    };
    let all_versions = conn
        .run(move |client| get_changelog_state(client, &project).unwrap())
        .await;
    Json(all_versions)
}

#[get("/project/<name>/version")]
async fn project_versions(conn: APICacheConnection, name: String) -> Json<Vec<Version>> {
    let proj_name = if name.is_ascii() {
        name.to_ascii_lowercase()
    } else {
        println!("invalid characters");
        String::default()
    };
    let project = Project {
        name: proj_name,
        ..Default::default()
    };
    let all_versions = conn
        .run(move |client| get_versions_for_project(client, &project).unwrap())
        .await;
    Json(all_versions)
}

#[catch(404)]
// fn not_found(req: &Request<'_>) -> Json<Response> {
fn not_found() -> Json<Response> {
    Json(Response {
        ok: false,
        code: 404,
        message: "data not found",
    })
}

pub async fn start_rocket() -> Result<(), rocket::error::Error> {
    rocket::build()
        .attach(APICacheConnection::fairing())
        .mount(
            "/",
            routes![index, projects, project_versions, project_changelog],
        )
        .register("/", catchers![not_found])
        .launch()
        .await?;
    Ok(())
}
