use crate::core::models::{Project, StatefulChangelog, Version};
use anyhow::Result;
use rocket_sync_db_pools::{database, memcache};
use serde_json;

#[database("n3nx_wolfie_cache")]
pub struct APICacheConnection(memcache::Client);
pub type CacheConnection = memcache::Client;

pub fn get_all_projects(cache: &CacheConnection) -> Result<Vec<Project>> {
    let proj_list = match cache.get("all_projects")? {
        Some::<String>(proj_string) => serde_json::from_str(&proj_string[..])?,
        None => {
            println!("cache error: nothing stored for key `all_projects`");
            Default::default()
        }
    };
    Ok(proj_list)
}

pub fn set_all_projects(cache: &CacheConnection, proj: &Vec<Project>) -> Result<()> {
    let proj_string = serde_json::to_string(proj)?;
    cache.set("all_projects", proj_string, 0)?;
    Ok(())
}

pub fn get_versions_for_project(cache: &CacheConnection, proj: &Project) -> Result<Vec<Version>> {
    let version_key = format!("version_{}", proj.name);
    let versions_list = match cache.get(&version_key[..])? {
        Some::<String>(version_string) => serde_json::from_str(&version_string[..])?,
        None => {
            println!("cache error: nothing stored for key `{}`", version_key);
            Default::default()
        }
    };
    Ok(versions_list)
}

pub fn set_versions_for_project(
    cache: &CacheConnection,
    proj: &Project,
    version: &Vec<Version>,
) -> Result<()> {
    let version_string = serde_json::to_string(version)?;
    let version_key = format!("version_{}", proj.name);
    cache.set(&version_key[..], version_string, 0)?;
    Ok(())
}

pub fn set_changelog_state(
    cache: &CacheConnection,
    proj: &Project,
    statelog: &StatefulChangelog,
) -> Result<()> {
    let statelog_string = serde_json::to_string(statelog)?;
    let statelog_key = format!("changelog_{}", proj.name);
    cache.set(&statelog_key[..], statelog_string, 0)?;
    Ok(())
}

pub fn get_changelog_state(cache: &CacheConnection, proj: &Project) -> Result<StatefulChangelog> {
    let statelog_key = format!("changelog_{}", proj.name);
    let statelog_data = match cache.get(&statelog_key)? {
        Some::<String>(statelog_string) => serde_json::from_str(&statelog_string[..])?,
        None => {
            println!("cache error: nothing stored for key `{}`", statelog_key);
            Default::default()
        }
    };
    Ok(statelog_data)
}
