use crate::core::models::{Changelog, Project};
use crate::db::entities::prelude::*;
use anyhow::{bail, Result};
use sea_orm::prelude::*;
use sea_orm::{DatabaseConnection, Set};

pub async fn update_project(
    conn: &DatabaseConnection,
    project: &Project,
    oldproject: &Project,
) -> Result<()> {
    // let new_project = ProjectActiveModel {
    //     name: Set(project.name.clone()),
    //     description: Set(project.desc.clone()),
    //     url: Set(project.url.clone()),
    //     is_featured: Set(project.featured),
    //     emoji: Set(project.emoji.clone()),
    //     is_template: Set(project.template),
    //     ..Default::default()
    // };

    let find_old_project = ProjectEntity::find()
        .filter(ProjectColumn::Name.eq(oldproject.name.as_str()))
        .one(conn)
        .await?;

    if let Some(oldproject_model) = find_old_project {
        let mut oldproject_active_model: ProjectActiveModel = oldproject_model.into();
        oldproject_active_model = ProjectActiveModel {
            name: Set(project.name.clone()),
            description: Set(project.desc.clone()),
            url: Set(project.url.clone()),
            is_featured: Set(project.featured),
            emoji: Set(project.emoji.clone()),
            is_template: Set(project.template),
            ..oldproject_active_model
        };
        oldproject_active_model.update(conn).await?;
    } else {
        bail!("project could not be found")
    };

    // delete old project tags
    TagEntity::delete_many()
        .filter(TagColumn::ProjectId.eq(&oldproject.name[..]))
        .exec(conn)
        .await?;
    // add new project tags
    let mut new_tags_list: Vec<TagActiveModel> = Vec::new();
    for tag in project.tags.iter() {
        let newtag = TagActiveModel {
            name: Set(tag.clone()),
            project_id: Set(project.name.clone()),
            ..Default::default()
        };
        new_tags_list.push(newtag)
    }
    TagEntity::insert_many(new_tags_list).exec(conn).await?;

    // update default project branch
    // let new_branch = BranchActiveModel {
    //     name: Set(project.branch.clone()),
    //     project_id: Set(project.name.clone()),
    //     ..Default::default()
    // };
    let find_branch = BranchEntity::find()
        .filter(BranchColumn::ProjectId.eq(&project.name[..]))
        .one(conn)
        .await;
    if let Some(branch_model) = find_branch? {
        let mut branch_active_model: BranchActiveModel = branch_model.into();
        branch_active_model = BranchActiveModel {
            name: Set(project.branch.clone()),
            project_id: Set(project.name.clone()),
            ..branch_active_model
        };
        branch_active_model.update(conn).await?;
    } else {
        bail!("update error: branch for project not found")
    }
    // let branch_update = BranchEntity::update(new_branch)
    //     .filter(BranchColumn::ProjectId.eq(&project.name[..]))
    //     .exec(conn)
    //     .await?;

    Ok(())
}

pub async fn update_changelog(
    conn: &DatabaseConnection,
    changelog: &Changelog,
    oldchangelog_hash: &str,
) -> Result<()> {
    let find_changelog = ChangelogEntity::find()
        .filter(ChangelogColumn::ProjectId.eq(&changelog.project_name[..]))
        .one(conn)
        .await;
    // let prev_hash_id = if let Some(changelog_model) = find_changelog? {
    //     changelog_model.prev_hash
    // } else {
    //     bail!("changelog update error: changelog model not found")
    // };
    // update changelog
    let prev_hash_id = if let Some(changelog_model) = find_changelog? {
        let stored_prev_hash = changelog_model.prev_hash.clone();
        let mut changelog_active_model: ChangelogActiveModel = changelog_model.into();
        changelog_active_model = ChangelogActiveModel {
            project_id: Set(changelog.project_name.clone()),
            hash: Set(changelog.hash.clone()),
            prev_hash: Set(oldchangelog_hash.to_owned()),
            ..changelog_active_model
        };
        changelog_active_model.update(conn).await?;
        stored_prev_hash
    } else {
        bail!("changelog update error: changelog model not found")
    };

    //delete prev hash data
    let find_hash = HashEntity::find()
        .filter(HashColumn::Id.eq(prev_hash_id))
        .one(conn)
        .await;
    if let Some(hash_model) = find_hash? {
        hash_model.delete(conn).await?;
    } else {
        bail!("changelog update error: previous hash model not found")
    };

    // insert new hash data
    let new_hash_model = HashActiveModel {
        id: Set(changelog.hash.clone()),
        data: Set(changelog.data.clone()),
        ..Default::default()
    };
    HashEntity::insert(new_hash_model).exec(conn).await?;

    // // update changelog
    // if let Some(changelog_model) = find_changelog? {
    //     let mut changelog_active_model: ChangelogActiveModel = changelog_model.into();
    //     changelog_active_model = ChangelogActiveModel {
    //         project_id: Set(changelog.project_name.clone()),
    //         hash: Set(changelog.hash.clone()),
    //         prev_hash: Set(oldchangelog_hash.to_owned()),
    //         ..changelog_active_model
    //     };
    //     changelog_active_model.update(conn).await?;
    // } else {
    //     bail!("changelog update error: changelog not found")
    // }
    // let updated_changelog = ChangelogActiveModel {
    //     project_id: Set(proj_id),
    //     hash: Set(changelog.hash.clone()),
    //     prev_hash: Set(oldchangelog_hash.to_owned()),
    //     ..Default::default()
    // };
    // ChangelogEntity::update(updated_changelog)
    //     .filter(ChangelogColumn::Hash.eq(oldchangelog_hash))
    //     .exec(conn)
    //     .await?;
    Ok(())
}
