use crate::consts::DEFAULT_DB_PATH;
use crate::core::models::{Config, Project, Version};
use anyhow::Result;
use async_trait::async_trait;
use reqwest::Client;
use sea_orm::{Database, DatabaseConnection};

pub trait APISecret {}

pub struct Adapter<'a, T: SocialAPI<'a>> {
    conn: DatabaseConnection,
    client: &'a T,
    project: &'a Project,
    version: &'a Version,
}

impl<'a, T: SocialAPI<'a>> Adapter<'a, T> {
    pub fn get_conn(&self) -> &DatabaseConnection {
        &self.conn
    }
    pub fn get_client(&self) -> &'a T {
        &self.client
    }
    pub fn get_project(&self) -> &'a Project {
        &self.project
    }
    pub fn get_version(&self) -> &'a Version {
        &self.version
    }
}

#[async_trait]
pub trait Publisher<'a> {
    async fn new<T: Sync + SocialAPI<'a>>(
        config: &Config,
        client: &'a T,
        project: &'a Project,
        version: &'a Version,
    ) -> Result<Adapter<'a, T>> {
        let db_path = config.db_url.as_ref().unwrap_or(&DEFAULT_DB_PATH);
        Ok(Adapter {
            conn: Database::connect(db_path).await?,
            client,
            project,
            version,
        })
    }
    async fn publish(&self) -> Result<()>;
}

pub trait Builder {
    fn build(proj: &Project, version: &Version) -> String;
}

pub struct SocialBox<'a, T: APISecret> {
    client: Client,
    api: &'a T,
}

impl<'a, T: APISecret> SocialBox<'a, T> {
    pub fn get_client(&self) -> &Client {
        &self.client
    }
    pub fn get_api(&self) -> &'a T {
        &self.api
    }
}

#[async_trait]
pub trait SocialAPI<'a> {
    fn new<T: 'a + APISecret>(client: Client, api: &T) -> SocialBox<T> {
        SocialBox { client, api }
    }

    async fn send(&self, msg: String) -> Result<String>;
    async fn recv(&self) -> Result<String>;
    async fn get(&self, msgid: &str) -> Result<String>;
    async fn delete(&self, msgid: &str) -> Result<String>;
}
