use super::log::N3NBox;
use super::models::{Config, Project, StatefulChangelog};
use super::traits::{Publisher, SocialAPI};
use crate::api::discord::{DiscordAdapter, DiscordBox};
use crate::api::telegram::{TelegramAdapter, TelegramBox};
use crate::api::twitter::{TwitterAdapter, TwitterBox};
use crate::cache::memory;
use crate::consts::{DEFAULT_DB_PATH, DEFAULT_SCAN_PERIOD};
use crate::db::functions::delete::delete_project;
use crate::db::functions::insert::{
    insert_or_ignore_changelog, insert_or_ignore_project, insert_or_ignore_versions,
};
use crate::db::functions::read::{get_all_project, get_all_versions, get_changelog_hash_ids};
use crate::db::functions::update::{update_changelog, update_project};
use crate::db::schema::create_entities;
use anyhow::Result;
use sea_orm::{Database, DatabaseConnection};
use std::thread;
use std::time::Duration;

pub async fn create(config: &Config) -> Result<(DatabaseConnection, memory::CacheConnection)> {
    let db_path = config.db_url.as_ref().unwrap_or(&DEFAULT_DB_PATH);

    if std::fs::metadata(db_path).is_err() {
        let db_conn = Database::connect(db_path).await?;
        let mut mem_conn = memory::CacheConnection::new();
        create_entities(&db_conn).await?;
        let all_n3n_boxes = N3NBox::fetch().await?;

        for n3n in all_n3n_boxes.iter() {
            let project = n3n.get_project();
            // create_project(&conn, project)?;
            insert_or_ignore_project(&db_conn, project).await?;

            if let Some(changelog) = n3n.get_changelog() {
                let stateful_changelog = StatefulChangelog {
                    project_name: changelog.project_name.to_string(),
                    hash: changelog.hash.to_string(),
                    prev_hash: changelog.hash.to_string(),
                };
                memory::set_changelog_state(&mut mem_conn, &project, &stateful_changelog)?;
                // create_changelog(&conn, changelog)?;
                insert_or_ignore_changelog(&db_conn, changelog).await?;
            }
            if let Some(versions) = n3n.get_versions() {
                memory::set_versions_for_project(&mut mem_conn, &project, &versions)?;
                insert_or_ignore_versions(&db_conn, versions).await?;
            }
        }

        let all_projects = get_all_project(&db_conn).await?;
        memory::set_all_projects(&mut mem_conn, &all_projects)?;

        let conn_set = (db_conn, mem_conn);
        Ok(conn_set)
    } else {
        // TODO: consume old data from database into memory cache for perfomant comparisons
        let db_conn = Database::connect(db_path).await?;
        let mut mem_conn = memory::CacheConnection::new();

        let all_projects = get_all_project(&db_conn).await?;
        memory::set_all_projects(&mut mem_conn, &all_projects)?;

        for project in all_projects.iter() {
            let (prev_hash, hash) = get_changelog_hash_ids(&db_conn, project).await?;
            let stateful_changelog = StatefulChangelog {
                project_name: project.name.to_string(),
                hash,
                prev_hash,
            };
            memory::set_changelog_state(&mut mem_conn, &project, &stateful_changelog)?;

            let versions = get_all_versions(&db_conn, &project).await?;
            memory::set_versions_for_project(&mut mem_conn, &project, &versions)?;
        }

        let conn_set = (db_conn, mem_conn);
        Ok(conn_set)
    }
}

async fn delete(db_conn: &DatabaseConnection, fetch_boxes: &Vec<N3NBox>) -> Result<()> {
    let upd_projects = get_all_project(db_conn).await?;
    let all_n3n_projects = fetch_boxes
        .iter()
        .map(|n3n_box| n3n_box.get_project())
        .collect::<Vec<&Project>>();
    for project in upd_projects.iter() {
        if !all_n3n_projects.contains(&project) {
            delete_project(db_conn, project).await?;
        }
    }
    Ok(())
}

pub async fn start_sync(config: &Config) -> Result<()> {
    let (conn, mem_conn) = create(config).await?;
    loop {
        let all_n3n_boxes = N3NBox::fetch().await?;
        // let old_projects = get_all_project(&conn).await?;
        let old_projects = memory::get_all_projects(&mem_conn)?;
        let old_proj_len = old_projects.len();
        for n3n_box in all_n3n_boxes.iter() {
            let now_proj = n3n_box.get_project();
            // Create project if not present already
            for (idx, old_proj) in old_projects.iter().enumerate() {
                match old_proj.name == now_proj.name {
                    true if old_proj != now_proj => {
                        update_project(&conn, now_proj, old_proj).await?
                    }
                    true => break,
                    false if idx == old_proj_len - 1 => {
                        insert_or_ignore_project(&conn, now_proj).await?
                    }
                    false => continue,
                }
            }
            // store changelog if not already
            if let Some(changelog) = n3n_box.get_changelog() {
                let now_hash = &changelog.hash;
                // let hashes_wrapped = get_changelog_hash_ids(&conn, now_proj).await;
                let old_hash_wrapped = memory::get_changelog_state(&mem_conn, now_proj);
                if old_hash_wrapped.is_err() {
                    insert_or_ignore_changelog(&conn, &changelog).await?;
                } else {
                    let old_hash = &old_hash_wrapped.as_ref().unwrap().prev_hash;
                    // let old_hash = hashes_wrapped?.0;
                    if now_hash != old_hash {
                        update_changelog(&conn, &changelog, old_hash).await?;
                    }
                }
            }
            // parse versions if not already
            if let Some(versions) = n3n_box.get_versions() {
                insert_or_ignore_versions(&conn, versions).await?;
            }
        }
        delete(&conn, &all_n3n_boxes).await?;

        post_on_social_media(&config, &all_n3n_boxes).await?;

        let scan_period = 60 * &config.scan_period.unwrap_or(DEFAULT_SCAN_PERIOD);
        thread::sleep(Duration::new(scan_period, 0));
    }
}

async fn post_on_social_media(config: &Config, fetch_boxes: &Vec<N3NBox>) -> Result<()> {
    let twitter_client = TwitterBox::new(reqwest::Client::new(), &config.twitter);
    let telegram_client = TelegramBox::new(reqwest::Client::new(), &config.telegram);
    let discord_client = DiscordBox::new(reqwest::Client::new(), &config.discord);

    for n3n_box in fetch_boxes.iter() {
        if let Some(versions) = n3n_box.get_versions() {
            for version in versions.iter() {
                let twitter_adapter =
                    TwitterAdapter::new(&config, &twitter_client, n3n_box.get_project(), version)
                        .await?;
                let discord_adapter =
                    DiscordAdapter::new(&config, &discord_client, n3n_box.get_project(), version)
                        .await?;
                let telegram_adapter =
                    TelegramAdapter::new(&config, &telegram_client, n3n_box.get_project(), version)
                        .await?;
                // twitter_adapter.publish()?;
                discord_adapter.publish().await?;
                // telegram_adapter.publish()?;
            }
        }
    }
    Ok(())
}
