use super::models::{AllGithubRepos, Changelog, GithubRepo, Project, ProjectMeta, Version};
use crate::consts::GITHUB_REPO_API;
use crate::utils::*;
use anyhow::Result;
use regex::Regex;
use reqwest;
use reqwest::header::{HeaderMap, USER_AGENT};

#[derive(Debug)]
pub struct N3NBox {
    project: Project,
    changelog: Option<Changelog>,
    versions: Option<Vec<Version>>,
}

impl N3NBox {
    pub async fn fetch() -> Result<Vec<N3NBox>> {
        let mut n3n_boxes = Vec::new();
        let github_repos = fetch_repos().await?;
        if !github_repos.incomplete_results {
            for repo in github_repos.items.into_iter() {
                let project = fetch_project(repo).await?;
                let changelog = fetch_changelog(&project).await?;
                let versions = fetch_version(&project, &changelog)?;
                n3n_boxes.push(N3NBox {
                    project,
                    changelog,
                    versions,
                })
            }
        } else {
        }
        Ok(n3n_boxes)
    }

    pub fn get_project(&self) -> &Project {
        &self.project
    }

    pub fn get_changelog(&self) -> &Option<Changelog> {
        &self.changelog
    }

    pub fn get_versions(&self) -> &Option<Vec<Version>> {
        &self.versions
    }
}

async fn fetch_repos() -> Result<AllGithubRepos> {
    let client = reqwest::Client::new();
    let mut heads = HeaderMap::new();
    heads.insert(
        USER_AGENT,
        "Mozilla/5.0 (Windows NT 10.0; rv:91.0) Gecko/20100101 Firefox/91.0".parse()?,
    );
    let resp = client.get(GITHUB_REPO_API).headers(heads).send().await?;
    let all_repos: AllGithubRepos = resp.json().await?;
    Ok(all_repos)
}

async fn fetch_project(repo: GithubRepo) -> Result<Project> {
    let client = reqwest::Client::new();
    let request_url = format!(
        "https://raw.githubusercontent.com/n3nx/{}/{}/README.md",
        repo.name, repo.default_branch
    );
    let response = client.get(&request_url).send().await?;
    let readme_raw: String = response.text().await?;
    let project_language = repo
        .language
        .as_ref()
        .unwrap_or(&String::from("template"))
        .to_string();
    if readme_raw.find("404:").is_some() {
        let emoji = emoji_from_proj_desc(&repo.description);
        return Ok(Project {
            name: repo.name,
            desc: repo.description,
            url: repo.html_url,
            tags: vec![project_language],
            featured: false,
            emoji,
            template: true,
            branch: repo.default_branch,
        });
        // bail!("changelog not found for project `{}`", project.name);
    };
    let project_meta = filter_readme_metadata(&repo, readme_raw);
    let project = Project {
        name: repo.name,
        desc: repo.description,
        url: repo.html_url,
        tags: project_meta.tags,
        featured: project_meta.featured,
        emoji: project_meta.emoji,
        template: project_meta.template,
        branch: repo.default_branch,
    };
    Ok(project)
}

pub fn filter_readme_metadata(repo: &GithubRepo, data: String) -> ProjectMeta {
    let project_language = repo
        .language
        .as_ref()
        .unwrap_or(&String::from("template"))
        .to_string();
    let project_meta_head = Regex::new(r"[<!\s-][Pp]?roject\s[Mm]?etadata[->\s]?").unwrap();
    if project_meta_head.is_match(&data) {
        let project_tags = Regex::new(r"project_tags:\s*([\w,\s]+)").unwrap();
        let project_featured = Regex::new(r"project_featured:\s*(true)").unwrap();
        let project_emoji = Regex::new(r"#\s([^\x00-\x7F\ufe0f]+)\s").unwrap();

        let emoji = if let Some(emoji) = project_emoji.captures(&data) {
            emoji.get(1).unwrap().as_str().to_string()
        } else {
            emoji_from_proj_desc(&repo.description)
        };

        let tags: Vec<String> = if let Some(t) = project_tags.captures(&data) {
            t.get(1)
                .unwrap()
                .as_str()
                .split(|c| c == ' ' || c == ',')
                .filter_map(|w| {
                    if w.is_empty() {
                        None
                    } else {
                        Some(w.to_string())
                    }
                })
                .collect()
        } else {
            vec![project_language]
        };
        let featured = project_featured.is_match(&data);
        let template = tags.contains(&"template".to_string());

        ProjectMeta {
            emoji,
            featured,
            template,
            tags,
        }
    } else {
        ProjectMeta {
            tags: vec![project_language],
            featured: false,
            emoji: "🐺".to_string(),
            template: true,
        }
    }
}

async fn fetch_changelog(project: &Project) -> Result<Option<Changelog>> {
    let client = reqwest::Client::new();
    if !project.template {
        let request_url = format!(
            "https://raw.githubusercontent.com/n3nx/{}/{}/CHANGELOG.md",
            project.name, project.branch
        );
        let response = client.get(&request_url).send().await?;
        let changelog_data_raw: String = response.text().await?;
        if changelog_data_raw.find("404:").is_some() {
            return Ok(None);
            // bail!("changelog not found for project `{}`", project.name);
        };
        let changelog_data = clean_changelog(&changelog_data_raw);
        let changelog_hash = gen_hash(&changelog_data);

        Ok(Some(Changelog {
            project_name: project.name.to_string(),
            data: changelog_data,
            hash: changelog_hash,
        }))
    } else {
        Ok(None)
    }
}

fn fetch_version(project: &Project, changelog: &Option<Changelog>) -> Result<Option<Vec<Version>>> {
    let mut all_versions: Vec<Version> = Vec::new();
    if let Some(valid_changelog) = changelog {
        let changelog_data = &valid_changelog.data[..];
        let split_marker = Regex::new(r"<!-.+\n").unwrap();
        for slice in split_marker
            .split(changelog_data)
            .filter_map(|x| match x.is_empty() {
                true => None,
                false => Some(x.trim()),
            })
        {
            let (version, version_num) = build_version(slice).unwrap();
            let (start_commit, end_commit) = build_commit_range(slice).unwrap();
            let date = build_date(slice).unwrap();
            let hash_id = gen_hash(slice);
            let version = Version {
                project_name: project.name.to_string(),
                hash: hash_id,
                data: slice.to_string(),
                timestamp: date.timestamp(),
                name: version,
                major: version_num[0],
                minor: version_num[1],
                patch: version_num[2],
                codename: String::new(),
                start_commit: start_commit,
                end_commit: end_commit,
            };
            all_versions.push(version)
        }
        Ok(Some(all_versions))
    } else {
        Ok(None)
    }
}
