use crate::core::models::{Project, StatefulChangelog, Version};
use anyhow::{bail, Result};
use serde_json;
use std::collections::BTreeMap;

pub type CacheConnection = BTreeMap<String, String>;

pub fn get_all_projects(cache: &CacheConnection) -> Result<Vec<Project>> {
    let proj_list = match cache.get("all_projects") {
        Some(proj_string) => serde_json::from_str(proj_string)?,
        None => bail!("cache error: nothing stored for key `all_projects`"),
    };
    Ok(proj_list)
}

pub fn set_all_projects(cache: &mut CacheConnection, proj: &Vec<Project>) -> Result<()> {
    let proj_string = serde_json::to_string(proj)?;
    cache.insert("all_projects".to_string(), proj_string);
    Ok(())
}

// pub fn get_versions_for_project(cache: &CacheConnection, proj: &Project) -> Result<Vec<Version>> {
//     let version_key = format!("version_{}", proj.name);
//     let versions_list = match cache.get(&version_key[..]) {
//         Some(version_string) => serde_json::from_str(version_string)?,
//         None => bail!("cache error: nothing stored for key `{}`", version_key),
//     };
//     Ok(versions_list)
// }

pub fn set_versions_for_project(
    cache: &mut CacheConnection,
    proj: &Project,
    version: &Vec<Version>,
) -> Result<()> {
    let version_string = serde_json::to_string(version)?;
    let version_key = format!("version_{}", proj.name);
    cache.insert(version_key, version_string);
    Ok(())
}

pub fn set_changelog_state(
    cache: &mut CacheConnection,
    proj: &Project,
    statelog: &StatefulChangelog,
) -> Result<()> {
    let statelog_string = serde_json::to_string(statelog)?;
    let statelog_key = format!("changelog_{}", proj.name);
    cache.insert(statelog_key, statelog_string);
    Ok(())
}

pub fn get_changelog_state(cache: &CacheConnection, proj: &Project) -> Result<StatefulChangelog> {
    let statelog_key = format!("changelog_{}", proj.name);
    let statelog_data = match cache.get(&statelog_key) {
        Some(statelog_string) => serde_json::from_str(statelog_string)?,
        None => bail!("cache error: nothing stored for key `{}`", statelog_key),
    };
    Ok(statelog_data)
}
