use crate::core::models::{Changelog, Project};
use crate::db::entities::prelude::*;
use anyhow::{bail, Result};
use sea_orm::prelude::*;
use sea_orm::{DatabaseConnection, Set};

pub async fn update_project(
    conn: &DatabaseConnection,
    project: &Project,
    oldproject: &Project,
) -> Result<()> {
    let new_project = ProjectActiveModel {
        name: Set(project.name.clone()),
        description: Set(project.desc.clone()),
        url: Set(project.url.clone()),
        is_featured: Set(project.featured),
        emoji: Set(project.emoji.clone()),
        is_template: Set(project.template),
        ..Default::default()
    };

    let oldproj_id = ProjectEntity::find()
        .filter(ProjectColumn::Name.eq(oldproject.name.as_str()))
        .one(conn)
        .await?
        .unwrap()
        .id;

    let project_update = ProjectEntity::update(new_project)
        .filter(ProjectColumn::Name.eq(oldproject.name.as_str()))
        .exec(conn)
        .await?;

    let newproj_id = ProjectEntity::find()
        .filter(ProjectColumn::Name.eq(project.name.as_str()))
        .one(conn)
        .await?
        .unwrap()
        .id;

    let new_branch = BranchActiveModel {
        name: Set(project.branch.clone()),
        project_id: Set(newproj_id),
        ..Default::default()
    };

    // delete old project tags
    TagEntity::delete_many()
        .filter(TagColumn::ProjectId.eq(oldproj_id))
        .exec(conn)
        .await?;

    // add new project tags
    let mut newproj_tags_vec: Vec<TagActiveModel> = Vec::new();
    for tag in project.tags.iter() {
        let proj_tag = TagActiveModel {
            name: Set(tag.clone()),
            project_id: Set(newproj_id),
            ..Default::default()
        };
        newproj_tags_vec.push(proj_tag)
    }
    TagEntity::insert_many(newproj_tags_vec).exec(conn).await;

    // update default project branch
    let branch_update = BranchEntity::update(new_branch)
        .filter(BranchColumn::ProjectId.eq(newproj_id))
        .exec(conn)
        .await?;

    Ok(())
}

pub async fn update_changelog(
    conn: &DatabaseConnection,
    changelog: &Changelog,
    oldchangelog_hash: &str,
) -> Result<()> {
    let proj_id = ProjectEntity::find()
        .filter(ProjectColumn::Name.eq(changelog.project_name.as_str()))
        .one(conn)
        .await?
        .unwrap()
        .id;
    let oldhash_id_wrapped = ChangelogEntity::find()
        .filter(ChangelogColumn::ProjectId.eq(proj_id))
        .one(conn)
        .await;
    let oldhash_id = if let Some(oldhash_id) = oldhash_id_wrapped? {
        oldhash_id.prev_hash
    } else {
        bail!("changelog previous hash not found")
    };
    let oldhash_data_wrapped = HashEntity::find()
        .filter(HashColumn::Id.eq(oldhash_id))
        .one(conn)
        .await;

    if let Some(oldhash_data) = oldhash_data_wrapped? {
        //delete old hash data
        oldhash_data.delete(conn).await?;
    } else {
        bail!("old hash data not found")
    };

    let newhash_data = HashActiveModel {
        id: Set(changelog.hash.clone()),
        data: Set(changelog.data.clone()),
        ..Default::default()
    };

    // insert new hash data
    HashEntity::insert(newhash_data).exec(conn).await?;

    // update changelog
    let updated_changelog = ChangelogActiveModel {
        project_id: Set(proj_id),
        hash: Set(changelog.hash.clone()),
        prev_hash: Set(oldchangelog_hash.to_owned()),
        ..Default::default()
    };
    ChangelogEntity::update(updated_changelog)
        .filter(ChangelogColumn::Hash.eq(oldchangelog_hash))
        .exec(conn)
        .await?;
    Ok(())
}
