use crate::core::models::{MediaId, Message, Project, Version};
use crate::db::entities::prelude::*;
use anyhow::{bail, Result};
use sea_orm::prelude::*;
use sea_orm::DatabaseConnection;

pub async fn get_all_project(conn: &DatabaseConnection) -> Result<Vec<Project>> {
    let all_project_rows = ProjectEntity::find().all(conn).await?;
    let mut all_project_vec: Vec<Project> = Vec::new();
    for row in all_project_rows.iter() {
        let project_data = Project {
            name: row.name.to_owned(),
            desc: row.description.to_owned(),
            url: row.url.to_owned(),
            emoji: row.emoji.to_owned(),
            featured: row.is_featured,
            template: row.is_template,
            tags: get_all_tags(conn, row.id).await?,
            branch: get_branch(conn, row.id).await?,
        };
        all_project_vec.push(project_data);
    }
    Ok(all_project_vec)
}

async fn get_branch(conn: &DatabaseConnection, proj_id: u32) -> Result<String> {
    let branch_row = TagEntity::find_by_id(proj_id).one(conn).await?;
    let branch_name = match branch_row {
        Some(m) => m.name.to_owned(),
        None => String::from("main"),
    };
    Ok(branch_name)
}

async fn get_all_tags(conn: &DatabaseConnection, proj_id: u32) -> Result<Vec<String>> {
    let all_tag_rows = TagEntity::find_by_id(proj_id).all(conn).await?;
    let all_tag_vec: Vec<String> = all_tag_rows.iter().map(|row| row.name.to_owned()).collect();
    Ok(all_tag_vec)
}

pub async fn get_changelog_hash_ids(
    conn: &DatabaseConnection,
    project: &Project,
) -> Result<(String, String)> {
    let proj_id = ProjectEntity::find()
        .filter(ProjectColumn::Name.contains(project.name.as_str()))
        .one(conn)
        .await?
        .unwrap()
        .id;

    let find_changelog = ChangelogEntity::find_by_id(proj_id).one(conn).await;
    let hash_ids = if let Some(changelog_model) = find_changelog? {
        let prev_hash = changelog_model.prev_hash;
        let hash = changelog_model.hash;
        (prev_hash, hash)
    } else {
        bail!("changelog for this project not found")
    };

    Ok(hash_ids)
}

async fn get_hash_data(conn: &DatabaseConnection, hash_id: String) -> Result<String> {
    let find_hash = HashEntity::find_by_id(hash_id).one(conn).await;
    let hash_data = if let Some(hash_model) = find_hash? {
        hash_model.data
    } else {
        bail!("hash data not found for given id")
    };
    Ok(hash_data)
}

pub async fn get_all_version_hashes(
    conn: &DatabaseConnection,
    project: &Project,
) -> Result<Vec<String>> {
    let proj_id = ProjectEntity::find()
        .filter(ProjectColumn::Name.eq(project.name.as_str()))
        .one(conn)
        .await?
        .unwrap()
        .id;
    let find_versions = VersionEntity::find()
        .filter(VersionColumn::ProjectId.eq(proj_id))
        .all(conn)
        .await;
    let mut all_version_hashes: Vec<String> = Vec::new();

    for v in find_versions?.iter() {
        all_version_hashes.push(v.hash.to_owned())
    }

    Ok(all_version_hashes)
}

pub async fn get_all_versions(
    conn: &DatabaseConnection,
    project: &Project,
) -> Result<Vec<Version>> {
    let proj_id = ProjectEntity::find()
        .filter(ProjectColumn::Name.eq(project.name.as_str()))
        .one(conn)
        .await?
        .unwrap()
        .id;
    let find_versions = VersionEntity::find()
        .filter(VersionColumn::ProjectId.eq(proj_id))
        .all(conn)
        .await;
    let mut all_version_data: Vec<Version> = Vec::new();

    for v in find_versions?.iter() {
        let version_data = Version {
            major: v.major,
            minor: v.minor,
            patch: v.patch,
            project_name: project.name.to_owned(),
            hash: v.hash.to_owned(),
            data: get_hash_data(conn, v.hash.to_owned()).await?,
            timestamp: v.timestamp.timestamp(),
            name: v.name.to_owned(),
            codename: v.codename.to_owned().unwrap_or_default(),
            start_commit: v.start_commit.to_owned().unwrap_or_default(),
            end_commit: v.end_commit.to_owned().unwrap_or_default(),
        };
        all_version_data.push(version_data)
    }

    Ok(all_version_data)
}

pub async fn get_message(
    conn: &DatabaseConnection,
    media: MediaId,
    version: &Version,
) -> Result<Message> {
    let platform_name: &str = media.into();
    let sns_id = SocialMediaEntity::find()
        .filter(SocialMediaColumn::PlatformName.eq(platform_name))
        .one(conn)
        .await?
        .unwrap()
        .id;
    let find_message = MessageEntity::find()
        .filter(SocialMediaColumn::Id.eq(sns_id))
        .filter(MessageColumn::VersionHash.eq(version.hash.as_str()))
        .one(conn)
        .await;
    let message_data = match find_message? {
        Some(m) => Message {
            media_id: media,
            project_name: version.project_name.clone(),
            version_hash: version.hash.clone(),
            timestamp: m.timestamp.timestamp(),
            message_id: m.message_id,
            message_hash: m.message_hash,
        },
        None => bail!("sent message for given version and platform not found"),
    };
    Ok(message_data)
}
