use crate::core::models::{Changelog, Message, Project, Version};
use crate::db::entities::prelude::*;
use anyhow::{bail, Result};
use chrono::prelude::*;
use sea_orm::prelude::*;
use sea_orm::{DatabaseConnection, Set};

pub async fn insert_or_ignore_project(conn: &DatabaseConnection, proj: &Project) -> Result<()> {
    // if project exists do not overwrite the entry
    let find_project = ProjectEntity::find()
        .filter(ProjectColumn::Name.contains(proj.name.as_str()))
        .one(conn)
        .await;
    match find_project? {
        Some(project_model) => println!("found project: {:?}, ignoring...", project_model),
        None => insert_project(conn, proj).await?,
    };
    Ok(())
}
async fn insert_project(conn: &DatabaseConnection, proj: &Project) -> Result<()> {
    let project_data = ProjectActiveModel {
        name: Set(proj.name.clone()),
        description: Set(proj.desc.clone()),
        url: Set(proj.url.clone()),
        emoji: Set(proj.emoji.clone()),
        is_featured: Set(proj.featured),
        is_template: Set(proj.template),
        ..Default::default()
    };

    ProjectEntity::insert(project_data).exec(conn).await?;

    let find_project = ProjectEntity::find()
        .filter(ProjectColumn::Name.contains(proj.name.as_str()))
        .one(conn)
        .await?;
    let proj_id = if let Some(project_model) = find_project {
        project_model.id
    } else {
        bail!("project {:?} not found in database", proj)
    };

    // Add project tags
    insert_tags(conn, proj, proj_id).await?;
    // Add project branch list
    insert_branch(conn, proj, proj_id).await?;

    Ok(())
}

async fn insert_tags(conn: &DatabaseConnection, proj: &Project, proj_id: u32) -> Result<()> {
    let mut tags_model_list: Vec<TagActiveModel> = Vec::new();
    for tag in proj.tags.iter() {
        let tag_data = TagActiveModel {
            name: Set(tag.clone()),
            project_id: Set(proj_id),
            ..Default::default()
        };
        tags_model_list.push(tag_data);
    }
    TagEntity::insert_many(tags_model_list).exec(conn).await?;
    Ok(())
}

async fn insert_branch(conn: &DatabaseConnection, proj: &Project, proj_id: u32) -> Result<()> {
    let branch_data = BranchActiveModel {
        name: Set(proj.branch.clone()),
        project_id: Set(proj_id),
        ..Default::default()
    };
    BranchEntity::insert(branch_data).exec(conn).await?;
    Ok(())
}

pub async fn insert_or_ignore_changelog(
    conn: &DatabaseConnection,
    changelog: &Changelog,
) -> Result<()> {
    // if project exists do not overwrite the entry
    let find_project = ProjectEntity::find()
        .filter(ProjectColumn::Name.contains(changelog.project_name.as_str()))
        .one(conn)
        .await?;
    let proj_id = match find_project {
        Some(project_model) => project_model.id,
        None => bail!("project {:?} not found in database", find_project),
    };
    let find_changelog = ChangelogEntity::find()
        .filter(ChangelogColumn::ProjectId.eq(proj_id))
        .one(conn)
        .await;
    match find_changelog? {
        Some(changelog_model) => println!("found changelog {:?}, ignoring...", changelog_model),
        None => insert_changelog(conn, changelog).await?,
    };
    Ok(())
}

async fn insert_changelog(conn: &DatabaseConnection, changelog: &Changelog) -> Result<()> {
    // Insert hash data
    let hash_data = HashActiveModel {
        id: Set(changelog.hash.clone()),
        data: Set(changelog.data.clone()),
        ..Default::default()
    };
    HashEntity::insert(hash_data).exec(conn).await?;

    // find project id
    let find_project = ProjectEntity::find()
        .filter(ProjectColumn::Name.contains(changelog.project_name.as_str()))
        .one(conn)
        .await?;
    let proj_id = if let Some(project_model) = find_project {
        project_model.id
    } else {
        bail!("project {:?} not found in database", find_project)
    };

    // Insert changelog data
    let changelog_data = ChangelogActiveModel {
        project_id: Set(proj_id),
        hash: Set(changelog.hash.clone()),
        prev_hash: Set(changelog.hash.clone()),
        ..Default::default()
    };
    ChangelogEntity::insert(changelog_data).exec(conn).await?;

    Ok(())
}

pub async fn insert_version(conn: &DatabaseConnection, version: &Version) -> Result<()> {
    let hash_data = HashActiveModel {
        id: Set(version.hash.clone()),
        data: Set(version.data.clone()),
        ..Default::default()
    };
    HashEntity::insert(hash_data).exec(conn).await?;

    // find project id
    let find_project = ProjectEntity::find()
        .filter(ProjectColumn::Name.contains(version.project_name.as_str()))
        .one(conn)
        .await?;
    let proj_id = if let Some(project_model) = find_project {
        project_model.id
    } else {
        bail!("project {:?} not found in database", find_project)
    };

    let version_data = VersionActiveModel {
        project_id: Set(proj_id),
        hash: Set(version.hash.clone()),
        timestamp: Set(chrono::Utc.timestamp(version.timestamp, 0)),
        name: Set(version.name.clone()),
        major: Set(version.major),
        minor: Set(version.minor),
        patch: Set(version.patch),
        // codename: Set(Some(version.codename.clone())),
        codename: Set(None),
        start_commit: Set(Some(version.start_commit.clone())),
        end_commit: Set(Some(version.end_commit.clone())),
    };
    VersionEntity::insert(version_data).exec(conn).await?;
    Ok(())
}

pub async fn insert_message(conn: &DatabaseConnection, message: &Message) -> Result<()> {
    let platform_name: &str = message.media_id.into();
    // find project id
    let find_sns = SocialMediaEntity::find()
        .filter(SocialMediaColumn::PlatformName.contains(platform_name))
        .one(conn)
        .await?;
    let sns_id = if let Some(sns_model) = find_sns {
        sns_model.id
    } else {
        bail!("social media {:?} not found in database", find_sns)
    };
    // find project id
    let find_project = ProjectEntity::find()
        .filter(ProjectColumn::Name.contains(message.project_name.as_str()))
        .one(conn)
        .await?;
    let proj_id = if let Some(project_model) = find_project {
        project_model.id
    } else {
        bail!("project {:?} not found in database", find_project)
    };

    let message_data = MessageActiveModel {
        social_media_id: Set(sns_id),
        project_id: Set(proj_id),
        version_hash: Set(message.version_hash.clone()),
        // timestamp: Set(message.timestamp),
        timestamp: Set(chrono::Utc.timestamp(message.timestamp, 0)),
        message_id: Set(message.message_id.clone()),
        message_hash: Set(message.message_hash.clone()),
    };
    MessageEntity::insert(message_data).exec(conn).await?;

    Ok(())
}
