use crate::core::models::{MediaId, Message, Project};
use crate::db::entities::prelude::*;
use anyhow::{bail, Result};
use sea_orm::prelude::*;
use sea_orm::DatabaseConnection;

async fn delete_version_hash(conn: &DatabaseConnection, project_id: u32) -> Result<()> {
    let find_version_hash = VersionEntity::find()
        .filter(VersionColumn::ProjectId.eq(project_id))
        .one(conn)
        .await;
    if let Some(version_model) = find_version_hash? {
        HashEntity::delete_many()
            .filter(HashColumn::Id.eq(version_model.hash))
            .exec(conn)
            .await?;
    } else {
        bail!("hash id not found")
    };
    Ok(())
}

async fn delete_all_versions(conn: &DatabaseConnection, project_id: u32) -> Result<()> {
    VersionEntity::delete_many()
        .filter(VersionColumn::ProjectId.eq(project_id))
        .exec(conn)
        .await?;
    Ok(())
}

async fn delete_logs(conn: &DatabaseConnection, project_id: u32) -> Result<()> {
    let find_changelog_model = ChangelogEntity::find()
        .filter(ChangelogColumn::ProjectId.eq(project_id))
        .one(conn)
        .await;
    let (prev_hash_model, next_hash_model) = if let Some(changelog_model) = find_changelog_model? {
        let prev_hash_model = HashEntity::find()
            .filter(HashColumn::Id.eq(changelog_model.prev_hash))
            .one(conn)
            .await;
        let next_hash_model = HashEntity::find()
            .filter(HashColumn::Id.eq(changelog_model.hash))
            .one(conn)
            .await;
        (prev_hash_model, next_hash_model)
    } else {
        bail!("changelog model not found")
    };

    if let Some(hash_model) = prev_hash_model? {
        hash_model.delete(conn).await?;
    } else {
        bail!("prev hash model not found");
    }

    if let Some(hash_model) = next_hash_model? {
        hash_model.delete(conn).await?;
    } else {
        bail!("prev hash model not found");
    }
    Ok(())
}

async fn delete_changelog(conn: &DatabaseConnection, project_id: u32) -> Result<()> {
    let find_changelog = ChangelogEntity::find()
        .filter(ChangelogColumn::ProjectId.eq(project_id))
        .one(conn)
        .await;
    if let Some(changelog_model) = find_changelog? {
        changelog_model.delete(conn).await?;
    } else {
        bail!("changelog model not found")
    };
    Ok(())
}

async fn delete_branch(conn: &DatabaseConnection, project_id: u32) -> Result<()> {
    let find_branch = BranchEntity::find()
        .filter(BranchColumn::ProjectId.eq(project_id))
        .one(conn)
        .await;
    if let Some(branch_model) = find_branch? {
        branch_model.delete(conn).await?;
    } else {
        bail!("branch model not found")
    };
    Ok(())
}

async fn delete_tags(conn: &DatabaseConnection, project_id: u32) -> Result<()> {
    TagEntity::delete_many()
        .filter(TagColumn::ProjectId.eq(project_id))
        .exec(conn)
        .await?;
    Ok(())
}

pub async fn delete_project(conn: &DatabaseConnection, project: &Project) -> Result<()> {
    let find_project = ProjectEntity::find()
        .filter(ProjectColumn::Name.eq(project.name.as_str()))
        .one(conn)
        .await;
    let proj_model = if let Some(proj_model) = find_project? {
        proj_model
    } else {
        bail!("project not found")
    };

    let proj_id = proj_model.id;

    // Delete all version hashes
    delete_version_hash(conn, proj_id).await?;

    // delete all versions
    delete_all_versions(conn, proj_id).await?;

    // delete logs
    delete_logs(conn, proj_id).await?;

    // delete changelog
    delete_changelog(conn, proj_id).await?;

    // delete branch
    delete_branch(conn, proj_id).await?;

    // delete tags
    delete_tags(conn, proj_id).await?;

    // delete project
    proj_model.delete(conn).await?;

    Ok(())
}

pub async fn delete_message(
    conn: &DatabaseConnection,
    media: MediaId,
    message: &Message,
) -> Result<()> {
    let platform_name: &str = media.into();

    let find_sns_model = SocialMediaEntity::find()
        .filter(SocialMediaColumn::PlatformName.eq(platform_name))
        .one(conn)
        .await;
    let sns_id = if let Some(sns_model) = find_sns_model? {
        sns_model.id
    } else {
        bail!("social media model not found")
    };
    let find_message = MessageEntity::find()
        .filter(MessageColumn::SocialMediaId.eq(sns_id))
        .filter(MessageColumn::MessageId.eq(message.message_id.as_str()))
        .one(conn)
        .await;
    if let Some(message_model) = find_message? {
        message_model.delete(conn).await?;
    } else {
        bail!("message not found")
    }
    Ok(())
}
