use crate::core::models::{Project, StatefulChangelog, Version};
use anyhow::Result;
use memcache;
use rkv::backend::{Lmdb, LmdbDatabase, LmdbEnvironment};
use rkv::{Manager, Rkv, SingleStore, StoreOptions, Value};
use serde_json;
use std::convert::TryFrom;
use std::fs;
use std::sync::{Arc, RwLock, RwLockReadGuard};
use tempfile::Builder;

pub struct CacheConnection {
    // env: RwLockReadGuard<'a, Rkv<LmdbEnvironment>>,
    created_arc: Arc<RwLock<Rkv<LmdbEnvironment>>>,
    // store: SingleStore<LmdbDatabase>,
}

impl CacheConnection {
    pub fn new() -> Result<CacheConnection, Box<dyn std::error::Error>> {
        let root = Builder::new().prefix("wolfie-cache").tempdir()?;
        fs::create_dir_all(root.path())?;
        let path = root.path();
        let mut manager = Manager::<LmdbEnvironment>::singleton().write()?;
        let created_arc = manager.get_or_create(path, Rkv::new::<Lmdb>)?;
        // let env = created_arc.read().unwrap();
        // let store = env.open_single("wolfiedb", StoreOptions::create())?;
        // let created_arc = manager.get_or_create(path, Rkv::new::<Lmdb>)?;
        // Ok(CacheConnection { created_arc, store })
        Ok(CacheConnection { created_arc })
    }

    //TODO: stop the repeated generations and consume created_arc only once
    pub fn get_store(&self) -> SingleStore<LmdbDatabase> {
        // &self.store
        let env = self.get_env();
        let store = env.open_single("wolfiedb", StoreOptions::create()).unwrap();
        store
    }

    pub fn get_env(&self) -> RwLockReadGuard<Rkv<LmdbEnvironment>> {
        self.created_arc.read().unwrap()
        // &self.env
    }
}

// pub fn get_connection() -> Result<memcache::Client> {
// let client = memcache::connect("memcache://127.0.0.1:11211?timeout=10&tcp_nodelay=true")?;
// client.flush()?;
// Ok(client)
// }

pub fn set_all_projects(client: &CacheConnection, proj: &Vec<Project>) {
    let project_string = serde_json::to_string(proj).unwrap();
    let env = client.get_env();
    let mut writer = env.write().unwrap();
    client
        .get_store()
        .put(
            &mut writer,
            "all_projects",
            &Value::Json(project_string.as_str()),
        )
        .unwrap();
    writer.commit().unwrap();
}

// pub fn get_all_projects(client: &memcache::Client) -> Result<Vec<Project>> {
pub fn get_all_projects(client: &CacheConnection) -> Result<Vec<Project>> {
    // let projects_string: Option<String> = client.get("all_projects").unwrap();
    let env = client.get_env();
    let reader = env.read().expect("reader");
    let projects_raw = client.get_store().get(&reader, "all_projects").unwrap();
    let projects_string = value_to_str(&projects_raw).unwrap();
    let projects: Vec<Project> = serde_json::from_str(projects_string).unwrap();
    Ok(projects)
}

pub fn value_to_str<'v>(value: &Option<Value<'v>>) -> Result<&'v str, &'v str> {
    let value = match value {
        Some(v) => v,
        None => return Err("no value obtained"),
    };
    match value {
        Value::Str(v) => Ok(v),
        _ => Err("type error: string value not found"),
    }
}

// pub fn set_versions(client: &memcache::Client, proj: &Project, versions: &Vec<Version>) {
pub fn set_versions(client: &CacheConnection, proj: &Project, versions: &Vec<Version>) {
    let versions_string = serde_json::to_string(versions).unwrap();
    let version_key = format!("version_{}", proj.name);
    // client.set(&version_key, versions_string, 0).unwrap();
    let env = client.get_env();
    let mut writer = env.write().unwrap();
    client
        .get_store()
        .put(&mut writer, &version_key, &Value::Json(&versions_string))
        .unwrap();
    writer.commit().unwrap();
}

// pub fn get_versions(client: &memcache::Client, proj: &Project) -> Result<Vec<Version>> {
pub fn get_versions(client: &CacheConnection, proj: &Project) -> Result<Vec<Version>> {
    // let projects_string = value_to_str(&projects_raw).unwrap();
    // let projects: Vec<Project> = serde_json::from_str(projects_string).unwrap();
    // Ok(projects)

    let version_key = format!("version_{}", proj.name);
    // let versions_string: Option<String> = client.get(&version_key).unwrap();
    let env = client.get_env();
    let reader = env.read().expect("reader");
    let versions_raw = client.get_store().get(&reader, &version_key).unwrap();
    let versions_string = value_to_str(&versions_raw).unwrap();
    let versions: Vec<Version> = serde_json::from_str(versions_string).unwrap();
    Ok(versions)
}

// pub fn set_changelog_state(
//     client: &memcache::Client,
//     proj: &Project,
//     changelog: &StatefulChangelog,
// ) {
pub fn set_changelog_state(
    client: &CacheConnection,
    proj: &Project,
    changelog: &StatefulChangelog,
) {
    let changelog_string = serde_json::to_string(changelog).unwrap();
    let changelog_key = format!("changelog_{}", proj.name);
    // client.set(&changelog_key, changelog_string, 0).unwrap();
    let env = client.get_env();
    let mut writer = env.write().unwrap();
    client
        .get_store()
        .put(&mut writer, &changelog_key, &Value::Json(&changelog_string))
        .unwrap();
    writer.commit().unwrap();
}

// pub fn get_changelog_state(client: &memcache::Client, proj: &Project) -> Result<StatefulChangelog> {
pub fn get_changelog_state(client: &CacheConnection, proj: &Project) -> Result<StatefulChangelog> {
    let changelog_key = format!("changelog_{}", proj.name);
    // let changelog_string: Option<String> = client.get(&changelog_key).unwrap();
    let env = client.get_env();
    let reader = env.read().expect("reader");
    let changelog_raw = client.get_store().get(&reader, &changelog_key).unwrap();
    let changelog_string = value_to_str(&changelog_raw).unwrap();
    let changelog: StatefulChangelog = serde_json::from_str(changelog_string).unwrap();
    Ok(changelog)
}
