use crate::core::models::{ConfigTwitter, MediaId, Message, Project, Version};
use crate::core::traits::{APISecret, Adapter, Builder, Publisher, SocialAPI, SocialBox};
use crate::db::functions::{delete::delete_message, insert::insert_message, read::get_message};
use crate::utils::gen_hash;
use chrono::prelude::*;
// use regex::Regex;
use anyhow::Result;
use async_trait::async_trait;
use oauth1_header::{http::Method, Credentials};
use reqwest::header::{HeaderMap, AUTHORIZATION, CONTENT_TYPE};
use sea_orm::DatabaseConnection;
use serde::{Deserialize, Serialize};
use std::collections::HashMap;

/***************************/
/*-------------------------*/
/*--                     --*/
/*--       Models        --*/
/*--                     --*/
/*-------------------------*/
/***************************/
#[derive(Debug, Serialize)]
pub struct TwitterSend {
    pub text: String,
}

#[derive(Deserialize)]
pub struct TwitterMessage {
    pub data: TwitterMessageData,
}

#[derive(Deserialize)]
pub struct TwitterMessageData {
    pub id: String,
    pub text: String,
}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--        Types        --*/
/*--                     --*/
/*-------------------------*/
/***************************/
pub type TwitterBox<'a> = SocialBox<'a, ConfigTwitter>;
pub type TwitterAdapter<'a> = Adapter<'a, TwitterBox<'a>>;
impl APISecret for ConfigTwitter {}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--     Social Box      --*/
/*--                     --*/
/*-------------------------*/
/***************************/
impl<'a> SocialAPI<'a> for TwitterBox<'a> {
    #[tokio::main]
    async fn send(&self, msg: String) -> Result<String> {
        let url = format!("{}/2/tweets", self.get_api().base_url);
        let request = TwitterSend { text: msg };
        let auth_header = self.gen_headers(&url, &Method::POST);
        let mut heads = HeaderMap::new();
        heads.insert(CONTENT_TYPE, "application/json".parse()?);
        heads.insert(AUTHORIZATION, auth_header.parse()?);
        let response = self
            .get_client()
            .post(&url)
            .json(&request)
            .headers(heads)
            .send()
            .await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }

    #[tokio::main]
    async fn get(&self, msgid: &str) -> Result<String> {
        let url = format!("{}/2/tweets/{}", self.get_api().base_url, msgid);
        let auth_header = self.gen_headers(&url, &Method::GET);
        let mut heads = HeaderMap::new();
        heads.insert(CONTENT_TYPE, "application/json".parse()?);
        heads.insert(AUTHORIZATION, auth_header.parse()?);
        let response = self.get_client().get(&url).headers(heads).send().await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }

    #[tokio::main]
    async fn delete(&self, msgid: &str) -> Result<String> {
        let url = format!("{}/2/tweets/{}", self.get_api().base_url, msgid);
        let mut heads = HeaderMap::new();
        let auth_header = self.gen_headers(&url, &Method::DELETE);
        // heads.insert(CONTENT_TYPE, "application/json".parse()?);
        heads.insert(AUTHORIZATION, auth_header.parse()?);
        let response = self.get_client().delete(&url).headers(heads).send().await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }
}

impl<'a> TwitterBox<'a> {
    fn gen_headers(&self, url: &str, method: &Method) -> String {
        let client = Credentials::new(
            &self.get_api().api_key,
            &self.get_api().api_secret,
            &self.get_api().access_token,
            &self.get_api().token_secret,
        );
        let params = HashMap::new();
        client.auth(method, &url, &params)
    }
}
/***************************/
/*-------------------------*/
/*--                     --*/
/*--      Builders       --*/
/*--                     --*/
/*-------------------------*/
/***************************/
impl<'a> Builder for TwitterBox<'a> {
    fn build(proj: &Project, version: &Version) -> String {
        version
            .data
            .lines()
            .map(|line| twitter_post_data(proj, version, line))
            .collect::<String>()
    }
}

fn twitter_post_data(proj: &Project, version: &Version, line: &str) -> String {
    if line.find("## ") == Some(0) {
        let date = Utc.timestamp(version.timestamp, 0);
        let tweet = format!(
            "{} {} v{}\n🎉 release {}-{}-{}\n🤩 repo at {}\n📜 changelog below",
            proj.emoji,
            proj.name,
            version.name,
            date.year(),
            date.month(),
            date.day(),
            proj.url
        );
        let tweet = build_tags(proj, tweet);
        format!(
            "{}\nhttps://n3n.org/news/{}-{}-{}-{}",
            tweet, proj.name, version.major, version.minor, version.patch
        )
    } else {
        String::new()
    }
}

fn build_tags(proj: &Project, line: String) -> String {
    let additional_tags = [
        "TrustIsUs",
        "cybersecurity",
        "infosec",
        "foss",
        "release",
        "opensource",
    ];
    let mut string = format!("{}\n", line);
    for tag in additional_tags.iter() {
        string = format!("{}#{} ", string, tag);
    }
    for tag in proj.tags.iter() {
        string = format!("{}#{} ", string, tag);
    }
    string
}
/***************************/
/*-------------------------*/
/*--                     --*/
/*--      Adapters       --*/
/*--                     --*/
/*-------------------------*/
/***************************/
#[async_trait]
impl<'a> Publisher<'a> for TwitterAdapter<'a> {
    async fn publish(&self) -> Result<()> {
        let stored_message =
            get_message(&self.get_conn(), MediaId::Twitter, &self.get_version()).await;
        if stored_message.is_err() {
            twitter_publish(
                &self.get_conn(),
                &self.get_client(),
                &self.get_project(),
                &self.get_version(),
            )
            .await?;
        } else {
            let stored_message = stored_message?;
            let message_resp = self.get_client().get(&stored_message.message_id)?;
            if message_resp.find("Not Found Error").is_some() {
                delete_message(&self.get_conn(), MediaId::Twitter, &stored_message).await?;
                twitter_publish(
                    &self.get_conn(),
                    &self.get_client(),
                    &self.get_project(),
                    &self.get_version(),
                )
                .await?;
            } else {
                let got_message: TwitterMessage = serde_json::from_str(&message_resp)?;
                let got_message_hash = gen_hash(&got_message.data.text);
                if stored_message.message_hash != got_message_hash {
                    twitter_publish(
                        &self.get_conn(),
                        &self.get_client(),
                        &self.get_project(),
                        &self.get_version(),
                    )
                    .await?;
                }
                // println!("{:#?}", stored_message);
            }
        }
        Ok(())
    }
}

async fn twitter_publish<'e>(
    conn: &DatabaseConnection,
    client: &TwitterBox<'e>,
    project: &Project,
    version: &Version,
) -> Result<()> {
    let twitter_response = client.send(TwitterBox::build(project, version))?;
    let message: TwitterMessage = serde_json::from_str(&twitter_response)?;
    let sent_message = Message {
        media_id: MediaId::Twitter,
        message_hash: gen_hash(&message.data.text),
        message_id: message.data.id,
        project_name: project.name.clone(),
        timestamp: Utc::now().timestamp(),
        version_hash: version.hash.clone(),
    };
    insert_message(conn, &sent_message).await?;
    Ok(())
}
